/*  Sesame - Storage and Querying architecture for RDF and RDF Schema
 *  Copyright (C) 2001-2007 Aduna
 *
 *  Contact:
 *  	Aduna
 *  	Prinses Julianaplein 14 b
 *  	3817 CS Amersfoort
 *  	The Netherlands
 *  	tel. +33 (0)33 465 99 87
 *  	fax. +33 (0)33 465 99 87
 *
 *  	http://aduna-software.com/
 *  	http://www.openrdf.org/
 *
 *  This library is free software; you can redistribute it and/or
 *  modify it under the terms of the GNU Lesser General Public
 *  License as published by the Free Software Foundation; either
 *  version 2.1 of the License, or (at your option) any later version.
 *
 *  This library is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 *  Lesser General Public License for more details.
 *
 *  You should have received a copy of the GNU Lesser General Public
 *  License along with this library; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

package org.openrdf.sesame.repository.remote;

import java.io.BufferedInputStream;
import java.io.File;
import java.io.FileInputStream;
import java.io.FileNotFoundException;
import java.io.IOException;
import java.io.InputStream;
import java.io.Reader;
import java.io.StringWriter;
import java.io.UnsupportedEncodingException;
import java.net.HttpURLConnection;
import java.net.MalformedURLException;
import java.net.URL;
import java.util.HashMap;
import java.util.Map;

import org.xml.sax.SAXException;

import org.openrdf.util.http.CookieManager;
import org.openrdf.util.http.HttpClientUtil;
import org.openrdf.util.io.IOUtil;
import org.openrdf.util.xml.XMLReaderFactory;

import org.openrdf.model.Graph;
import org.openrdf.model.Resource;
import org.openrdf.model.Statement;
import org.openrdf.model.URI;
import org.openrdf.model.Value;

import org.openrdf.rio.NamespaceListener;
import org.openrdf.rio.ParseErrorListener;
import org.openrdf.rio.ParseException;
import org.openrdf.rio.Parser;
import org.openrdf.rio.StatementHandler;
import org.openrdf.rio.StatementHandlerException;
import org.openrdf.rio.ntriples.NTriplesParser;
import org.openrdf.rio.ntriples.NTriplesUtil;
import org.openrdf.rio.rdfxml.RdfXmlParser;
import org.openrdf.rio.turtle.TurtleParser;
import org.openrdf.rio.turtle.TurtleWriter;

import org.openrdf.sesame.admin.AdminListener;
import org.openrdf.sesame.admin.AdminMsgCollector;
import org.openrdf.sesame.admin.XmlAdminMsgReader;
import org.openrdf.sesame.config.AccessDeniedException;
import org.openrdf.sesame.constants.AdminResultFormat;
import org.openrdf.sesame.constants.QueryLanguage;
import org.openrdf.sesame.constants.QueryResultFormat;
import org.openrdf.sesame.constants.RDFFormat;
import org.openrdf.sesame.query.BinaryTableResultReader;
import org.openrdf.sesame.query.GraphQueryResultListener;
import org.openrdf.sesame.query.MalformedQueryException;
import org.openrdf.sesame.query.QueryEvaluationException;
import org.openrdf.sesame.query.QueryResultsGraphBuilder;
import org.openrdf.sesame.query.QueryResultsTable;
import org.openrdf.sesame.query.QueryResultsTableBuilder;
import org.openrdf.sesame.query.TableQueryResultListener;
import org.openrdf.sesame.query.XmlQueryResultReader;
import org.openrdf.sesame.repository.SesameRepository;
import org.openrdf.sesame.sail.StatementIterator;

/**
 * A proxy for a single remote Sesame repository that can be reached over HTTP.
 * Create this object using an HTTPService.
 * 
 * @see HTTPService
 * 
 * @author Jeen Broekstra
 * @author Arjohn Kampman
 */
public class HTTPRepository implements SesameRepository {

/*----------+
| Constants |
+----------*/

	private static final String TABLE_QUERY_SERVLET = "servlets/evaluateTableQuery";
	private static final String GRAPH_QUERY_SERVLET = "servlets/evaluateGraphQuery";
	private static final String DATA_UPLOAD_SERVLET = "servlets/uploadData";
	private static final String DATA_MERGE_SERVLET = "servlets/mergeData";
	
	private static final String URL_UPLOAD_SERVLET = "servlets/uploadURL";
	private static final String GRAPH_QUERY_ADD_SERVLET = "servlets/addGraphQuery";
	private static final String GRAPH_QUERY_MERGE_SERVLET = "servlets/mergeGraphQuery";
	private static final String GRAPH_QUERY_REMOVE_SERVLET = "servlets/removeGraphQuery";
	private static final String REMOVE_STATS_SERVLET = "servlets/removeStatements";
	private static final String CLEAR_SERVLET = "servlets/clearRepository";
	private static final String EXTRACT_SERVLET = "servlets/extractRDF";

/*----------+
| Variables |
+----------*/

	/**
	 * The repository identifier for this repository.
	 **/
	private String _repositoryId;

	/** The response format for table query results. **/
	private QueryResultFormat _tableQueryResultFormat = QueryResultFormat.XML;

	/** The response format for graph query results. **/
	private RDFFormat _graphQueryResultFormat = RDFFormat.RDFXML;

	private CookieManager _cookieManager;

	// URLs for all of the available actions.
	private URL _tableQueryURL;
	private URL _graphQueryURL;
	private URL _dataUploadURL;
	private URL _urlUploadURL;
	private URL _dataMergeURL;
	private URL _removeStatsURL;
	private URL _clearURL;
	private URL _extractURL;
	private URL _addGraphQueryURL;
	private URL _mergeGraphQueryURL;
	private URL _removeGraphQueryURL;

/*-------------+
| Constructors |
+-------------*/

	/**
	 * Constructor is protected: create objects through the HTTPService.
	 * 
	 * @see HTTPService
	 **/
	protected HTTPRepository(URL serverURL, String repositoryId, CookieManager cookieManager)
		throws MalformedURLException
	{
		_repositoryId = repositoryId;
		_cookieManager = cookieManager;

		_tableQueryURL       = HTTPService.resolveURL(serverURL, TABLE_QUERY_SERVLET);
		_graphQueryURL       = HTTPService.resolveURL(serverURL, GRAPH_QUERY_SERVLET);
		_dataUploadURL       = HTTPService.resolveURL(serverURL, DATA_UPLOAD_SERVLET);
		_dataMergeURL        = HTTPService.resolveURL(serverURL, DATA_MERGE_SERVLET);
		_urlUploadURL        = HTTPService.resolveURL(serverURL, URL_UPLOAD_SERVLET);
		_removeStatsURL      = HTTPService.resolveURL(serverURL, REMOVE_STATS_SERVLET);
		_clearURL            = HTTPService.resolveURL(serverURL, CLEAR_SERVLET);
		_extractURL          = HTTPService.resolveURL(serverURL, EXTRACT_SERVLET);
		_addGraphQueryURL    = HTTPService.resolveURL(serverURL, GRAPH_QUERY_ADD_SERVLET);
		_mergeGraphQueryURL  = HTTPService.resolveURL(serverURL, GRAPH_QUERY_MERGE_SERVLET);
		_removeGraphQueryURL = HTTPService.resolveURL(serverURL, GRAPH_QUERY_REMOVE_SERVLET);
	}

/*--------+
| Methods |
+--------*/

	/**
	 * Sets the response format for table query results. The default format is
	 * <tt>QueryResultFormat.XML</tt>. Setting another format may result in
	 * better performance and/or a more compact result requiring less bandwidth.
	 *
	 * @param format A table query result format. Allowed values are <tt>XML</tt> and
	 * <tt>BINARY</tt>.
	 **/
	public void setTableQueryResultFormat(QueryResultFormat format) {
		_tableQueryResultFormat = format;
	}

	// implements SesameRepository.performTableQuery(QueryLanguage, String, TableQueryResultListener)
	public void performTableQuery(QueryLanguage language, String query, TableQueryResultListener listener)
		throws IOException, MalformedQueryException, QueryEvaluationException, AccessDeniedException
	{
		// Build the query string
		Map postParams = new HashMap();
		postParams.put("repository", _repositoryId);
		postParams.put("query", query);
		postParams.put("queryLanguage", language.toString());
		postParams.put("resultFormat", _tableQueryResultFormat.toString());

		InputStream resultStream = _sendPostRequest(_tableQueryURL, postParams);

		try {
			// Parse the result and feed it to the supplied listener
			if (_tableQueryResultFormat == QueryResultFormat.XML) {
				try {
					XmlQueryResultReader xmlReader = new XmlQueryResultReader(XMLReaderFactory.createXMLReader());
					xmlReader.read(resultStream, listener);
				}
				catch (SAXException e) {
					// SAXException wraps another exception?
					Exception wrappedExc = e.getException();
					String msg = (wrappedExc != null) ? wrappedExc.getMessage() : e.getMessage();
					_throwIOException("Unable to parse query results from Sesame: " + msg, e);
				}
			}
			else if (_tableQueryResultFormat == QueryResultFormat.BINARY) {
				BinaryTableResultReader binReader = new BinaryTableResultReader();
				binReader.read(resultStream, listener);
			}
			else {
				throw new RuntimeException("Unsupported table query result format: " + _tableQueryResultFormat.toString());
			}
		}
		finally {
			resultStream.close();
		}
	}

	// implements SesameRepository.performTableQuery(QueryLanguage, String)
	public QueryResultsTable performTableQuery(QueryLanguage language, String query)
		throws IOException, MalformedQueryException, QueryEvaluationException, AccessDeniedException
	{
		QueryResultsTableBuilder tableBuilder = new QueryResultsTableBuilder();

		performTableQuery(language, query, tableBuilder);

		return tableBuilder.getQueryResultsTable();
	}

	/**
	 * Sets the response format for graph query results. The default format is
	 * <tt>RDFFormat.RDFXML</tt>. Setting another format may result in better
	 * performance and/or a more compact result requiring less bandwidth.
	 *
	 * @param format An RDF serialization format. Allowed values are <tt>RDFXML</tt>,
	 * <tt>NTRIPLES</tt> and <tt>TURTLE</tt>.
	 **/
	public void setGraphQueryResultFormat(RDFFormat format) {
		_graphQueryResultFormat = format;
	}

	// implements SesameRepository.performGraphQuery(QueryLanguage, String, GraphQueryResultListener)
	public void performGraphQuery(QueryLanguage language, String query, GraphQueryResultListener listener)
		throws IOException, MalformedQueryException, QueryEvaluationException, AccessDeniedException
	{
		// Build the query string
		Map postParams = new HashMap();
		postParams.put("repository", _repositoryId);
		postParams.put("query", query);
		postParams.put("queryLanguage", language.toString());
		postParams.put("serialization", _graphQueryResultFormat.toString());

		// Create and configure parser
		Parser parser = null;
		if (_graphQueryResultFormat == RDFFormat.RDFXML) {
			parser = new RdfXmlParser();
		}
		else if (_graphQueryResultFormat == RDFFormat.NTRIPLES) {
			parser = new NTriplesParser();
		}
		else if (_graphQueryResultFormat == RDFFormat.TURTLE) {
			parser = new TurtleParser();
		}
		else {
			throw new RuntimeException("Unsupported graph query result format: " + _graphQueryResultFormat.toString());
		}

		RDFParserListener rdfParserListener = new RDFParserListener(listener);
		parser.setStatementHandler(rdfParserListener);
		parser.setParseErrorListener(rdfParserListener);
		parser.setNamespaceListener(rdfParserListener);

		parser.setVerifyData(false);
		parser.setPreserveBNodeIds(true);

		// Send the request and process the result
		InputStream resultStream = _sendPostRequest(_graphQueryURL, postParams);

		try {
			parser.parse(resultStream, "foo:bar");
			rdfParserListener.finishGraphQueryResult();
		}
		catch (StatementHandlerException e) {
			_throwIOException(e);
		}
		catch (ParseException e) {
			throw new QueryEvaluationException(e);
		}
		finally {
			resultStream.close();
		}
	}

	// implements SesameRepository.performGraphQuery(QueryLanguage, String)
	public Graph performGraphQuery(QueryLanguage language, String query)
		throws IOException, MalformedQueryException, QueryEvaluationException, AccessDeniedException
	{
		QueryResultsGraphBuilder listener = new QueryResultsGraphBuilder();
		
		performGraphQuery(language, query, listener);
		
		return listener.getGraph();
	}
	
	// implements SesameRepository.addData(URL, String, RDFFormat, boolean, AdminListener)
	public void addData(URL dataURL, String baseURI, RDFFormat dataFormat, boolean verifyData, AdminListener listener)
		throws IOException, AccessDeniedException
	{
		// Build the query string
		Map postParams = new HashMap();
		postParams.put("repository", _repositoryId);
		postParams.put("url", dataURL.toExternalForm());
		if (baseURI != null) {
			postParams.put("baseURI", baseURI);
		}
		postParams.put("dataFormat", dataFormat.toString());
		if (verifyData) {
			postParams.put("verifyData", "on");
		}
		postParams.put("resultFormat", AdminResultFormat.XML.toString());

		_adminPost(_urlUploadURL, postParams, listener);
	}

	// implements SesameRepository.addData(File, String, RDFFormat, boolean, AdminListener)
	public void addData(File dataFile, String baseURI, RDFFormat dataFormat, boolean verifyData, AdminListener listener)
		throws FileNotFoundException, IOException, AccessDeniedException
	{
		FileInputStream in = new FileInputStream(dataFile);
		try {
			addData(in, baseURI, dataFormat, verifyData, listener);
		}
		finally {
			in.close();
		}
	}

	// implements SesameRepository.addData(InputStream, String, RDFFormat, boolean, AdminListener)
	public void addData(InputStream data, String baseURI, RDFFormat dataFormat, boolean verifyData, AdminListener listener)
		throws IOException, AccessDeniedException
	{
		// Read the contents from InputStream 'data':
		_addData(data, baseURI, dataFormat, verifyData, listener);
	}

	// implements SesameRepository.addData(Reader, String, RDFFormat, boolean, AdminListener)
	public void addData(Reader data, String baseURI, RDFFormat dataFormat, boolean verifyData, AdminListener listener)
		throws IOException, AccessDeniedException
	{
		// Read the contents from Reader 'data':
		StringWriter stringWriter = new StringWriter(8192);
		IOUtil.transfer(data, stringWriter);

		// Upload it as a String:
		addData(stringWriter.toString(), baseURI, dataFormat, verifyData, listener);
	}

	// implements SesameRepository.addData(String, String, RDFFormat, boolean, AdminListener)
	public void addData(String data, String baseURI, RDFFormat dataFormat, boolean verifyData, AdminListener listener)
		throws IOException, AccessDeniedException
	{
		_addData(data, baseURI, dataFormat, verifyData, listener);
	}

	// implements SesameRepository.addData(SesameRepository, AdminListener)
	public void addData(SesameRepository repository, AdminListener listener)
		throws IOException, AccessDeniedException
	{
		if (this == repository) {
			// Adding all data from this to this is useless. This special case
			// is handled to prevent potential deadlocks.
			return;
		}

		// FIXME: more efficient implementation possible for some cases
		InputStream dataStream = repository.extractRDF(RDFFormat.TURTLE, true, true, true, false);
		try {
			addData(dataStream, null, RDFFormat.TURTLE, false, listener);
		}
		finally {
			dataStream.close();
		}
	}

	/**
	 * Methods for uploading RDF data to a server. The supplied <tt>data</tt>
	 * parameter should be either a String, a byte array, or a
	 * <tt>org.openrdf.util.http.FilePart</tt> object.
	 **/
	private void _addData(Object data, String baseURI, RDFFormat dataFormat, boolean verifyData, AdminListener listener)
		throws IOException, AccessDeniedException
	{
		// Build the query string
		Map postParams = new HashMap();
		postParams.put("repository", _repositoryId);
		postParams.put("data", data);
		postParams.put("dataFormat", dataFormat.toString());
		postParams.put("resultFormat", AdminResultFormat.XML.toString());
		if (baseURI != null) {
			postParams.put("baseURI", baseURI);
		}
		if (verifyData) {
			postParams.put("verifyData", "on");
		}

		_adminPost(_dataUploadURL, postParams, listener);
	}

	/* (non-Javadoc)
	 * @see org.openrdf.sesame.repository.SesameRepository#addGraph(org.openrdf.model.Graph)
	 */
	public void addGraph(Graph graph)
		throws IOException, AccessDeniedException
	{
		String turtleDoc = _graph2TurtleDoc(graph);

		// Build the query string
		Map postParams = new HashMap();
		postParams.put("repository", _repositoryId);
		postParams.put("data", turtleDoc);
		postParams.put("dataFormat", RDFFormat.TURTLE.toString());
		postParams.put("resultFormat", AdminResultFormat.XML.toString());
	
		AdminMsgCollector listener = new AdminMsgCollector();
		_adminPost(_dataMergeURL, postParams, listener);
		
	}

	/* (non-Javadoc)
	 * @see org.openrdf.sesame.repository.SesameRepository#addGraph(org.openrdf.sesame.constants.QueryLanguage, java.lang.String)
	 */
	public void addGraph(QueryLanguage language, String query)
		throws IOException, AccessDeniedException
	{
		// Build the query string
		Map postParams = new HashMap();
		postParams.put("repository", _repositoryId);
		postParams.put("query", query);
		postParams.put("queryLanguage", language.toString());
		postParams.put("resultFormat", AdminResultFormat.XML.toString());

		AdminMsgCollector listener = new AdminMsgCollector();

		_adminPost(_mergeGraphQueryURL, postParams, listener);
	}


	public void addGraph(Graph graph, boolean joinBlankNodes)
		throws IOException, AccessDeniedException
	{
		if (!joinBlankNodes) { 
			String turtleDoc = _graph2TurtleDoc(graph);
			
			AdminMsgCollector listener = new AdminMsgCollector();
			addData(turtleDoc, null, RDFFormat.TURTLE, false, listener);
			
			if (listener.hasErrors()) {
				// FIXME what to do with errors during upload?
			}
		}
		else {
			addGraph(graph);
		}
	}
	

	// SesameRepository.addGraph(QueryLanguage, String)
	public void addGraph(QueryLanguage language, String query, boolean joinBlankNodes)
	throws IOException, AccessDeniedException
	{
		if (!joinBlankNodes) {
			// Build the query string
			Map postParams = new HashMap();
			postParams.put("repository", _repositoryId);
			postParams.put("query", query);
			postParams.put("queryLanguage", language.toString());
			postParams.put("resultFormat", AdminResultFormat.XML.toString());
			
			AdminMsgCollector listener = new AdminMsgCollector();
			
			_adminPost(_addGraphQueryURL, postParams, listener);
		}
		else {
			addGraph(language, query);
		}
	}

	// implements SesameRepository.extractData(RDFFormat, boolean, boolean, boolean, boolean)
	public InputStream extractRDF(RDFFormat serialization, boolean schema, boolean data, boolean explicitOnly, boolean niceOutput)
		throws IOException, AccessDeniedException
	{
		// Build the query string
		Map postParams = new HashMap();
		postParams.put("repository", _repositoryId);
		postParams.put("serialization", serialization.toString());
		if (schema) {
			postParams.put("schema", "on");
		}
		if (data) {
			postParams.put("data", "on");
		}
		if (explicitOnly) {
			postParams.put("explicitOnly", "on");
		}
		if (niceOutput) {
			postParams.put("niceOutput", "on");
		}

		try {
			return _sendPostRequest(_extractURL, postParams);
		}
		catch (MalformedQueryException e) {
			// Never thrown in this context
			_throwIOException(e);
		}
		catch (QueryEvaluationException e) {
			// Never thrown in this context
			_throwIOException(e);
		}

		// Statement never reached but needed to keep the compiler happy.
		return null;
	}

	// implements SesameRepository.removeStatements(Resource, URI, Value, AdminListener)
	public void removeStatements(Resource subject, URI predicate, Value object, AdminListener listener)
		throws IOException, AccessDeniedException
	{
		// Build the query string
		Map postParams = new HashMap();
		postParams.put("repository", _repositoryId);
		postParams.put("resultFormat", AdminResultFormat.XML.toString());

		if (subject != null) {
			postParams.put("subject", NTriplesUtil.toNTriplesString(subject));
		}
		if (predicate != null) {
			postParams.put("predicate", NTriplesUtil.toNTriplesString(predicate));
		}
		if (object != null) {
			postParams.put("object", NTriplesUtil.toNTriplesString(object));
		}

		_adminPost(_removeStatsURL, postParams, listener);
	}


	// implements SesameRepository.removeGraph(Graph)
	public void removeGraph(Graph graph)
		throws IOException, AccessDeniedException
	{
		String turtleDoc = _graph2TurtleDoc(graph);

		// Build the query string
		Map postParams = new HashMap();
		postParams.put("repository", _repositoryId);
		postParams.put("dataFormat", RDFFormat.TURTLE.toString());
		postParams.put("resultFormat", AdminResultFormat.XML.toString());
		postParams.put("data", turtleDoc);

		AdminMsgCollector listener = new AdminMsgCollector();
		_adminPost(_removeStatsURL, postParams, listener);

		if (listener.hasErrors()) {
			// FIXME what to do with errors during upload?
		}
	}

	// SesameRepository.removeGraph(QueryLanguage, String)
	public void removeGraph(QueryLanguage language, String query)
		throws IOException, AccessDeniedException
	{
		// Build the query string
		Map postParams = new HashMap();
		postParams.put("repository", _repositoryId);
		postParams.put("query", query);
		postParams.put("queryLanguage", language.toString());
		postParams.put("resultFormat", AdminResultFormat.XML.toString());

		AdminMsgCollector listener = new AdminMsgCollector();

		_adminPost(_removeGraphQueryURL, postParams, listener);
	}

	// implements SesameRepository.clear(AdminListener)
	public void clear(AdminListener listener)
		throws IOException, AccessDeniedException
	{
		// Build the query string
		Map postParams = new HashMap();
		postParams.put("repository", _repositoryId);
		postParams.put("resultFormat", AdminResultFormat.XML.toString());

		_adminPost(_clearURL, postParams, listener);
	}
	
	public String getRepositoryId() {
		return _repositoryId;
	}

	/**
	 * Generates a Turtle document from the supplied graph.
	 **/
	private String _graph2TurtleDoc(Graph graph)
		throws IOException
	{
		StringWriter stringWriter = new StringWriter(8192);
		TurtleWriter turtleWriter = new TurtleWriter(stringWriter);

		turtleWriter.startDocument();

		StatementIterator statIter = graph.getStatements();
		while (statIter.hasNext()) {
			Statement st = statIter.next();
			turtleWriter.writeStatement(st.getSubject(), st.getPredicate(), st.getObject());
		}
		statIter.close();

		turtleWriter.endDocument();

		return stringWriter.toString();
	}

	/**
	 * Sends an HTTP-POST request with the supplied postData to the supplied
	 * URL. Status, warning and error messages will be reported to the supplied
	 * AdminListener.
	 */
	private void _adminPost(URL url, Map postParams, AdminListener listener)
		throws IOException, AccessDeniedException
	{
		InputStream responseStream = null;

		try {
			responseStream = _sendPostRequest(url, postParams);
		}
		catch (MalformedQueryException e) {
			// Never thrown in this context
			_throwIOException(e);
		}
		catch (QueryEvaluationException e) {
			// Never thrown in this context
			_throwIOException(e);
		}

		// Parse the response
		try {
			XmlAdminMsgReader reader = new XmlAdminMsgReader(XMLReaderFactory.createXMLReader());
			reader.read(responseStream, listener);
		}
		catch (SAXException e) {
			String msg;

			// Nested exception?
			Exception ne = e.getException();
			if (ne != null) {
				msg = ne.getMessage();
			}
			else {
				msg = e.getMessage();
			}

			_throwIOException("Unable to handle request: " + msg, e);
		}
		finally {
			responseStream.close();
		}
	}

	/**
	 * Sends an HTTP-POST request with the supplied postData to the supplied
	 * URL. The server's response is checked, throwing an exception when it
	 * indicates an error. Otherwise, the data returned by the server is
	 * returned as an InputStream.
	 *
	 * @param url The URL to post the data to.
	 * @param postParams The parameters for the POST request.
	 * @return An input stream containing the data that was returned by the server.
	 * @exception AccessDeniedException In case one does not have access to the
	 * specified resource.
	 * @exception MalformedQueryException In case the server was unable to parse
	 * the query that was sent to it.
	 * @exception QueryEvaluationException In case the server failed to evaluate
	 * the query that was sent to it.
	 * @exception IOException In case of an error in the request or in the
	 * processing of it by the server.
	 */
	private InputStream _sendPostRequest(URL url, Map postParams)
		throws IOException, AccessDeniedException, MalformedQueryException, QueryEvaluationException
	{
		// Set up request
		HttpURLConnection conn = (HttpURLConnection) url.openConnection();
		_cookieManager.setCookies(conn);
		HttpClientUtil.setAcceptGZIPEncoding(conn);
		try {
			HttpClientUtil.prepareMultipartPostRequestInputStreamAware(conn, postParams, "UTF-8");
		}
		catch (UnsupportedEncodingException e) {
			// UTF-8 must be supported by all compliant JVM's, this exception should never be thrown.
			throw new RuntimeException("UTF-8 character encoding not supported on this platform");
		}

		// Send the request
		conn.connect();

		// Check whether the server reported any errors
		_checkResponse(conn);

		// Get buffered input stream (HttpClientUtil takes care of any gzip compression)
		return new BufferedInputStream(HttpClientUtil.getInputStream(conn), 2048);
	}

	private void _checkResponse(HttpURLConnection conn)
		throws IOException, AccessDeniedException, MalformedQueryException, QueryEvaluationException
	{
		int responseCode = conn.getResponseCode();

		if (responseCode == HttpURLConnection.HTTP_OK) {
			return;
		}

		String responseMsg = conn.getResponseMessage();
		//System.out.println("Got error code: " + responseCode + " " + responseMsg);

		if (responseCode == HttpURLConnection.HTTP_FORBIDDEN) {
			throw new AccessDeniedException(responseMsg);
		}
		else if (responseCode == HttpURLConnection.HTTP_BAD_REQUEST && responseMsg != null) {
			int colonIdx = responseMsg.indexOf(':');
			if (colonIdx > 0) {
				HTTPErrorType errType = HTTPErrorType.forValue(responseMsg.substring(0, colonIdx));

				if (errType == HTTPErrorType.UNKNOWN_REPOSITORY) {
					// Repository removed on server?
					throw new IOException(responseMsg.substring(colonIdx + 2));
				}
				else if (errType == HTTPErrorType.MALFORMED_QUERY) {
					throw new MalformedQueryException(responseMsg.substring(colonIdx + 2));
				}
				else if (errType == HTTPErrorType.QUERY_EVALUATION_ERROR) {
					throw new QueryEvaluationException(responseMsg.substring(colonIdx + 2));
				}
			}
			else {
				throw new IOException(responseMsg);
			}
		}
		else {
			throw new IOException(responseMsg);
		}
	}

	private void _throwIOException(Exception e)
		throws IOException
	{
		IOException ioe = new IOException(e.getMessage());
		ioe.initCause(e);
		throw ioe;
	}

	private void _throwIOException(String msg, Exception e)
		throws IOException
	{
		IOException ioe = new IOException(msg);
		ioe.initCause(e);
		throw ioe;
	}

/*------------------------------+
| Inner class RDFParserListener |
+------------------------------*/

	/**
	 * A listener for various events from an RDF parser that reports these
	 * events to a GraphQueryResultListener.
	 */
	static class RDFParserListener
		implements StatementHandler, ParseErrorListener, NamespaceListener
	{
		/**
		 * The listener to report events to.
		 */
		private GraphQueryResultListener _listener;

		/**
		 * Flag indicating whether the start of the query result has already
		 * been reported to the listener.
		 */
		private boolean _queryResultStarted;

		/**
		 * Creates an RDFParserListener that reports events to the supplied
		 * listener.
		 */
		public RDFParserListener(GraphQueryResultListener listener) {
			_listener = listener;
			_queryResultStarted = false;
		}

		// Calling startGraphQueryResult() is deferred until all namespaces have
		// been reported
		private void _ensureQueryResultStarted() throws IOException {
			if (!_queryResultStarted) {
				_listener.startGraphQueryResult();
				_queryResultStarted = true;
			}
		}

		// Implements StatementHandler.handleStatement(...)
		public void handleStatement(Resource subject, URI predicate, Value object)
			throws StatementHandlerException {
			try {
				_ensureQueryResultStarted();
				_listener.triple(subject, predicate, object);
			}
			catch (IOException e) {
				throw new StatementHandlerException(e);
			}
		}

		// Implements ParseErrorListener.warning(...)
		public void warning(String msg, int lineNo, int colNo) {
			// ignore
		}

		// Implements ParseErrorListener.error(...)
		public void error(String msg, int lineNo, int colNo) {
			try {
				_ensureQueryResultStarted();
				_listener.reportError(msg);
			}
			catch (Exception e) {
				// FIXME ignore for now.
			}
		}

		// Implements ParseErrorListener.fatalError(...)
		public void fatalError(String msg, int lineNo, int colNo) {
			try {
				_ensureQueryResultStarted();
				_listener.reportError(msg);
			}
			catch (Exception e) {
				// FIXME ignore for now.
			}
		}

		// Implements NamespaceListener.handleNamespace(String, String)
		public void handleNamespace(String prefix, String uri) {
			try {
				_listener.namespace(prefix, uri);
			}
			catch (IOException e) {
				// ignore
			}
		}

		/**
		 * Reports the end of the graph query result to the
		 * GraphQueryResultListener. This method should be called when the RDF
		 * parser has finished parsing.
		 */
		public void finishGraphQueryResult() {
			try {
				_ensureQueryResultStarted();
				_listener.endGraphQueryResult();
			}
			catch (IOException e) {
				// FIXME ignore for now
			}
		}
	} // end inner class
}
