/*  Sesame - Storage and Querying architecture for RDF and RDF Schema
 *  Copyright (C) 2001-2006 Aduna
 *
 *  Contact: 
 *  	Aduna
 *  	Prinses Julianaplein 14 b
 *  	3817 CS Amersfoort
 *  	The Netherlands
 *  	tel. +33 (0)33 465 99 87
 *  	fax. +33 (0)33 465 99 87
 *
 *  	http://aduna-software.com/
 *  	http://www.openrdf.org/
 *  
 *  This library is free software; you can redistribute it and/or
 *  modify it under the terms of the GNU Lesser General Public
 *  License as published by the Free Software Foundation; either
 *  version 2.1 of the License, or (at your option) any later version.
 *
 *  This library is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 *  Lesser General Public License for more details.
 *
 *  You should have received a copy of the GNU Lesser General Public
 *  License along with this library; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

package org.openrdf.sesame.repository.remote;

import org.openrdf.util.Enumeration;

/**
 * A type-safe enumeration class for HTTP error types. Each error type has a
 * unique string value that is transfered over HTTP as a prefix for an error
 * message. The following is an example HTTP header indicating that an unknown
 * repository has been specified.
 * <p>
 * <tt>HTTP/1.1 400 UNKNOWN_REPOSITORY: repository 'testdb' not found</tt>
 *
 * @author Arjohn Kampman
 * @version $Revision: 1.2.4.2 $
 */
public final class HTTPErrorType extends Enumeration {

/*------------------+
| Static stuff      |
+------------------*/

	/**
	 * Constant indicating that an unknown repository was specified.
	 **/
	public static final HTTPErrorType UNKNOWN_REPOSITORY = new HTTPErrorType("UNKNOWN_REPOSITORY");

	/**
	 * Constant indicating that a malformed query has been specified.
	 **/
	public static final HTTPErrorType MALFORMED_QUERY = new HTTPErrorType("MALFORMED_QUERY");

	/**
	 * Constant indicating that a error occured during the evaluation of a query.
	 **/
	public static final HTTPErrorType QUERY_EVALUATION_ERROR = new HTTPErrorType("QUERY_EVALUATION_ERROR");

	/**
	 * An HTTPErrorType array containing all possible values.
	 **/
	private static HTTPErrorType[] _values =
			new HTTPErrorType[] {UNKNOWN_REPOSITORY, MALFORMED_QUERY, QUERY_EVALUATION_ERROR};

	/**
	 * Returns an HTTPErrorType constant for the specified value.
	 *
	 * @param value The String-representation of an HTTPErrorType constant.
	 * @return An HTTPErrorType object, or <tt>null</tt> if the specified
	 * value could not be resolved to an HTTPErrorType object.
	 **/
	public static HTTPErrorType forValue(String value) {
		return (HTTPErrorType)_forValue(_values, value);
	}

/*------------------+
| Non-static stuff  |
+------------------*/

	/**
	 * Creates a new HTTPErrorType object.
	 **/
	private HTTPErrorType(String value) {
		super(value);
	}
}
