/*  Sesame - Storage and Querying architecture for RDF and RDF Schema
 *  Copyright (C) 2001-2006 Aduna
 *
 *  Contact:
 *  	Aduna
 *  	Prinses Julianaplein 14 b
 *  	3817 CS Amersfoort
 *  	The Netherlands
 *  	tel. +33 (0)33 465 99 87
 *  	fax. +33 (0)33 465 99 87
 *
 *  	http://aduna-software.com/
 *  	http://www.openrdf.org/
 *
 *  This library is free software; you can redistribute it and/or
 *  modify it under the terms of the GNU Lesser General Public
 *  License as published by the Free Software Foundation; either
 *  version 2.1 of the License, or (at your option) any later version.
 *
 *  This library is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 *  Lesser General Public License for more details.
 *
 *  You should have received a copy of the GNU Lesser General Public
 *  License along with this library; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

package org.openrdf.sesame.repository.local;

import java.io.File;
import java.io.IOException;
import java.util.HashMap;
import java.util.Iterator;
import java.util.List;
import java.util.ListIterator;
import java.util.Map;

import org.openrdf.sesame.config.AccessDeniedException;
import org.openrdf.sesame.config.ConfigurationException;
import org.openrdf.sesame.config.RepositoryConfig;
import org.openrdf.sesame.config.RepositoryListImpl;
import org.openrdf.sesame.config.SailConfig;
import org.openrdf.sesame.config.SystemConfig;
import org.openrdf.sesame.config.UnknownRepositoryException;
import org.openrdf.sesame.config.UserInfo;
import org.openrdf.sesame.omm.SessionContext;
import org.openrdf.sesame.repository.RepositoryList;
import org.openrdf.sesame.repository.SesameRepository;
import org.openrdf.sesame.repository.SesameService;
import org.openrdf.sesame.sail.RdfSource;
import org.openrdf.sesame.sail.Sail;
import org.openrdf.sesame.sail.SailInitializationException;
import org.openrdf.sesame.sail.StackedSail;
import org.openrdf.sesame.server.SesameServer;

/**
 * A Sesame service for local repositories.
 *
 * @author Jeen Broekstra
 * @author Arjohn Kampman
 */
public class LocalService implements SesameService {

/*---------------------+
| Variables            |
+---------------------*/

	private SystemConfig _systemConfig;

	private Map _repositoryMap;

/*---------------------+
| Constructors         |
+---------------------*/

	/**
	 * Creates a new LocalService that does not administer any
	 * repositories yet.
	 **/
	public LocalService() {
		this(new SystemConfig());
	}

	/**
	 * Creates a new LocalService for all repositories defined in the
	 * supplied <tt>SystemConfig</tt> object.
	 *
	 * @param systemConfig A SystemConfig object containing repository
	 * definitions.
	 **/
	public LocalService(SystemConfig systemConfig) {
		_repositoryMap = new HashMap();
		setSystemConfig(systemConfig);
	}

/*---------------------+
| Methods              |
+---------------------*/

	/**
	 * Sets or updates the system configuration that is used by this
	 * service.
	 **/
	public void setSystemConfig(SystemConfig newConfig) {
		synchronized (_repositoryMap) {
			if (_systemConfig == null) {
				_systemConfig = new SystemConfig();
			}

			Map newRepositoryMap = new HashMap();
			Map oldRepositoryMap = new HashMap(_repositoryMap);

			Iterator iter = newConfig.getRepositoryConfigList().iterator();
			while (iter.hasNext()) {
				RepositoryConfig newRepConfig = (RepositoryConfig)iter.next();
				String id = newRepConfig.getRepositoryId();

				// If the 'new' RepositoryConfig hasn't changed, then reuse the
				// old SesameRepository
				RepositoryConfig oldRepConfig = _systemConfig.getRepositoryConfig(id);
				if (oldRepConfig != null && newRepConfig.equals(oldRepConfig)) {
					Object repository = oldRepositoryMap.remove(id);
					if (repository != null) {
						newRepositoryMap.put(id, repository);
					}
				}
			}

			_systemConfig = newConfig;
			_repositoryMap = newRepositoryMap;

			// Shut down all changed or removed repositories
			iter = oldRepositoryMap.values().iterator();
			while (iter.hasNext()) {
				LocalRepository expiredRep = (LocalRepository)iter.next();
				expiredRep.shutDown();
			}
		}
	}

	/**
	 * Gets the system configuration that is used by this service.
	 **/
	public SystemConfig getSystemConfig() {
		return _systemConfig;
	}

	/**
	 * Log in to a Sesame service. As the login process is Thread-based, the
	 * login will only apply to the thread executing this method.
	 *
	 * @exception AccessDeniedException If the attempt to log in failed.
	 * @exception IOException If an I/O error occurred.
	 * @exception IllegalArgumentException If the supplied username is
	 * not a legal username.
	 **/
	public void login(String user, String password)
		throws AccessDeniedException
	{
		SessionContext sc = _getSessionContext();

		UserInfo ui = _systemConfig.getUserInfo(user);
		if (ui == null) {
			throw new AccessDeniedException("User '" + user + "' unknown");
		}
		if (ui.getPassword() != null && !ui.getPassword().equals(password)) {
			throw new AccessDeniedException("Password incorrect");
		}

		sc.fullname = ui.getFullName();
		sc.user = ui.getLogin();
		sc.pass = ui.getPassword();
		sc.userID = ui.getID();
	}

	// implements SesameService.logout()
	public void logout() {
		SessionContext sc = _getSessionContext();
		sc.fullname = "";
		sc.user = "";
		sc.pass = "";
		sc.userID = 0;
	}

	// implements SesameService.getRepositoryList()
	public RepositoryList getRepositoryList() {
		synchronized (_systemConfig) {
			RepositoryListImpl repositoryList = new RepositoryListImpl();

			String username = _getSessionContext().user;
			UserInfo userInfo = _systemConfig.getUserInfo(username);

			List repConfigs = _systemConfig.getRepositoryConfigList();
			for (int i = 0; i < repConfigs.size(); i++) {
				RepositoryConfig repConfig = (RepositoryConfig)repConfigs.get(i);

				String repositoryId = repConfig.getRepositoryId();

				boolean readAccess = repConfig.isWorldReadable();
				boolean writeAccess = repConfig.isWorldWriteable();

				if (readAccess == false && userInfo != null) {
					// Not world readable but might be readable for this user
					readAccess = userInfo.hasReadAccess(repositoryId);
				}

				if (writeAccess == false && userInfo != null) {
					// Not world writeable but might be writeable for this user
					writeAccess = userInfo.hasWriteAccess(repositoryId);
				}

				repositoryList.addRepository(repositoryId, repConfig.getTitle(),
						readAccess, writeAccess);
			}

			return repositoryList;
		}
	}

	// implements SesameService.getRepository(String)
	public SesameRepository getRepository(String repositoryId)
		throws UnknownRepositoryException, ConfigurationException
	{
		synchronized (_repositoryMap) {
			LocalRepository repository = (LocalRepository)_repositoryMap.get(repositoryId);

			if (repository == null) {
				// First call, create the repository.
				synchronized (_systemConfig) {
					RepositoryConfig repConfig = _systemConfig.getRepositoryConfig(repositoryId);

					if (repConfig == null) {
						throw new UnknownRepositoryException("Unknown repository: " + repositoryId);
					}

					repository = _createRepository(repConfig);
					_repositoryMap.put(repositoryId, repository);
				}
			}

			return repository;
		}
	}

	/**
	 * Defines a new repository.
	 *
	 * @param repConfig The new repository's configuration.
	 * @exception ConfigurationException If a repository with an
	 * identical ID is already defined.
	 **/
	public void addRepository(RepositoryConfig repConfig)
		throws ConfigurationException
	{
		String repositoryId = repConfig.getRepositoryId();

		synchronized (_systemConfig) {
			if (_systemConfig.hasRepository(repositoryId)) {
				throw new ConfigurationException("A repository with ID '"
						+ repositoryId + "' is already defined");
			}

			_systemConfig.addRepositoryConfig(repConfig);
		}
	}

	/**
	 * Removes a repository definition.
	 *
	 * @param repositoryId The ID of the repository to remove.
	 **/
	public void removeRepository(String repositoryId) {
		synchronized (_systemConfig) {
			_systemConfig.removeRepository(repositoryId);
		}
		synchronized (_repositoryMap) {
			_repositoryMap.remove(repositoryId);
		}
	}

	/**
	 * Adds a new repository definition to this LocalService and
	 * creates a SesameRepository object for it.
	 *
	 * @param repConfig A repository configuration.
	 * @return A SesameRepository matching the supplied configuration.
	 * @exception ConfigurationException If the supplied configuration
	 * cannot be added for some reason.
	 **/
	public LocalRepository createRepository(RepositoryConfig repConfig)
		throws ConfigurationException
	{
		addRepository(repConfig);

		try {
			return (LocalRepository)getRepository(repConfig.getRepositoryId());
		}
		catch (UnknownRepositoryException e) {
			// This exception should never be thrown
			throw new RuntimeException("Program error", e);
		}
	}

	protected static SessionContext _getSessionContext() {
		SessionContext sc = SessionContext.getContext();

		if (sc == null) {
			sc = new SessionContext();
			sc.bLogged = false;
			sc.VersionState = -1;

			SessionContext.setContext(sc);
		}

		return sc;
	}
	
	/**
	 * Adds a new repository with the supplied ID to this LocalService and
	 * creates a LocalRepository object for it. The repository will be a world-readable/writeable, 
	 * synchronized in-memory repository, without persistence.
	 * 
	 * @param repositoryId the ID for this repository.
	 * @param inferencing indicates if the repository should be an inferencing repository.
	 * 
	 * @return A SesameRepository using the in-memory sail, and a synchronization layer.
	 * 
	 * @exception ConfigurationException If a repository with the supplied ID
	 * cannot be added for some reason, e.g. if the ID was already in use.
	 */
	public LocalRepository createRepository(String repositoryId, boolean inferencing) 
		throws ConfigurationException 
	{
		RepositoryConfig config = new RepositoryConfig(repositoryId);
		SailConfig syncSail;
		SailConfig memSail;
		
		if (inferencing) {
			syncSail = new SailConfig("org.openrdf.sesame.sailimpl.sync.SyncRdfSchemaRepository");
			memSail = new SailConfig("org.openrdf.sesame.sailimpl.memory.RdfSchemaRepository");
		}
		else {
			syncSail = new SailConfig("org.openrdf.sesame.sailimpl.sync.SyncRdfRepository");
			memSail = new SailConfig("org.openrdf.sesame.sailimpl.memory.RdfRepository");
		}

		config.addSail(syncSail);
		config.addSail(memSail);
		config.setWorldReadable(true);
		config.setWorldWriteable(true);
		
		return createRepository(config);
	}

	private LocalRepository _createRepository(RepositoryConfig repConfig)
		throws ConfigurationException
	{
		try {
			// Create a new Sail for the bottom of the Sail stack
			List sailList = repConfig.getSailList();
			ListIterator sailIter = sailList.listIterator(sailList.size());

			SailConfig sailConfig = (SailConfig)sailIter.previous();
			String className = sailConfig.getSailClass();

			Class sailClass = Class.forName(className);
			Sail topSail = null;
			try {
				topSail = (Sail)sailClass.newInstance();
			}
			catch (ClassCastException e) {
				throw new ConfigurationException(sailClass + " does not implement Sail interface.");
			}
			topSail.initialize(sailConfig.getConfigParameters());

			// Add any stacked Sails on top
			while (sailIter.hasPrevious()) {
				sailConfig = (SailConfig)sailIter.previous();

				className = sailConfig.getSailClass();
				sailClass = Class.forName(className);

				StackedSail stackedSail = null;
				try {
					stackedSail = (StackedSail)sailClass.newInstance();
				}
				catch (ClassCastException e) {
					throw new ConfigurationException(className + " is not a StackedSail.");
				}
				stackedSail.setBaseSail(topSail);
				stackedSail.initialize(sailConfig.getConfigParameters());

				topSail = stackedSail;
			}

			if (topSail instanceof RdfSource) {
				return new LocalRepository(repConfig.getRepositoryId(),
						(RdfSource)topSail, this);
			}
			else {
				throw new ConfigurationException(
						"top Sail class does not implement RdfSource");
			}
		}
		catch (ClassNotFoundException e) {
			throw new ConfigurationException(e);
		}
		catch (InstantiationException e) {
			throw new ConfigurationException(e);
		}
		catch (IllegalAccessException e) {
			throw new ConfigurationException(e);
		}
		catch (SailInitializationException e) {
			throw new ConfigurationException(e);
		}
	}

	/**
	 * Shuts down all repositories that are configured for this service.
	 **/
	public void shutDown() {
		synchronized (_repositoryMap) {
			Iterator iter = _repositoryMap.values().iterator();
			while (iter.hasNext()) {
				LocalRepository rep = (LocalRepository)iter.next();
				rep.shutDown();
			}
			_repositoryMap.clear();
		}
	}

	/**
	 * Checks whether the user that has logged in has read access on the
	 * specified repository. If no user has been logged in, this method will
	 * only return <tt>true</tt> if the repository is world-readable.
	 *
	 * @param repository A repository ID.
	 * @return <tt>true</tt> if the user has read access, <tt>false</tt>
	 * otherwise.
	 **/
	public boolean hasReadAccess(String repository)
		throws UnknownRepositoryException
	{
		RepositoryConfig repConfig = _systemConfig.getRepositoryConfig(repository);
		if (repConfig == null) {
			throw new UnknownRepositoryException("Unknown repository: " + repository);
		}

		if (repConfig.isWorldReadable()) {
			// repository is world-readable
			return true;
		}

		// Not world-readable, check if user has access
		String username = _getSessionContext().user;
		UserInfo userInfo = _systemConfig.getUserInfo(username);

		if (userInfo != null) {
			return userInfo.hasReadAccess(repConfig.getRepositoryId());
		}

		return false;
	}

	/**
	 * Checks whether the user that has logged in has write access on the
	 * specified repository. If no user has been logged in, this method will
	 * only return <tt>true</tt> if the repository is world-writeable.
	 *
	 * @param repository A repository ID.
	 * @return <tt>true</tt> if the user has write access, <tt>false</tt>
	 * otherwise.
	 **/
	public boolean hasWriteAccess(String repository)
		throws UnknownRepositoryException
	{
		RepositoryConfig repConfig = _systemConfig.getRepositoryConfig(repository);
		if (repConfig == null) {
			throw new UnknownRepositoryException("Unknown repository: " + repository);
		}

		if (repConfig.isWorldWriteable()) {
			// repository is world-writeable
			return true;
		}

		// Not world-writeable, check if user has access
		String username = _getSessionContext().user;
		UserInfo userInfo = _systemConfig.getUserInfo(username);

		if (userInfo != null) {
			return userInfo.hasWriteAccess(repConfig.getRepositoryId());
		}

		return false;
	}
	
	public File getTmpDir()
		throws IOException
	{
		String tmpDirStr = _systemConfig.getTmpDir();
		File tmpDir = null;
		
		if (tmpDirStr != null) {
			tmpDir = new File(SesameServer.getBaseDir(), tmpDirStr);
			
			if (!tmpDir.exists()) {
				boolean tmpDirCreated = tmpDir.mkdirs();
				
				if (!tmpDirCreated) {
					throw new IOException("Unable to create directory " + tmpDir.getAbsolutePath());
				}
			}
		}
		
		return tmpDir;
	}
	
	/**
	 * Creates a new file with the supplied prefix and suffix in the configured
	 * tmp directory.
	 *
	 * @param prefix The prefix string to be used in generating the file's
	 * name; must be at least three characters long.
	 * @param suffix The suffix string to be used in generating the file's
	 * name; may be null, in which case the suffix ".tmp" will be used
	 * @return A new tmp file, or null if no tmp directory has been configured.
	 * @throws IOException If an I/O error occured during the creation of the
	 * tmp file.
	 **/
	public File createTmpFile(String prefix, String suffix)
		throws IOException
	{
		File tmpDir = getTmpDir();
		File tmpFile = null;
		
		if (tmpDir != null) {
			tmpFile = File.createTempFile(prefix, suffix, tmpDir);
			// Note: do not use tmpFile.deleteOnExit() as this leaks
			// considerable amounts of memory. See for more information:
			// http://bugs.sun.com/bugdatabase/view_bug.do?bug_id=4513817
		}
		
		return tmpFile;
	}
}
