/*  Sesame - Storage and Querying architecture for RDF and RDF Schema
 *  Copyright (C) 2001-2006 Aduna
 *
 *  Contact: 
 *  	Aduna
 *  	Prinses Julianaplein 14 b
 *  	3817 CS Amersfoort
 *  	The Netherlands
 *  	tel. +33 (0)33 465 99 87
 *  	fax. +33 (0)33 465 99 87
 *
 *  	http://aduna-software.com/
 *  	http://www.openrdf.org/
 *  
 *  This library is free software; you can redistribute it and/or
 *  modify it under the terms of the GNU Lesser General Public
 *  License as published by the Free Software Foundation; either
 *  version 2.1 of the License, or (at your option) any later version.
 *
 *  This library is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 *  Lesser General Public License for more details.
 *
 *  You should have received a copy of the GNU Lesser General Public
 *  License along with this library; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

package org.openrdf.sesame.repository.local;

import java.io.ByteArrayInputStream;
import java.io.ByteArrayOutputStream;
import java.io.File;
import java.io.FileInputStream;
import java.io.FileNotFoundException;
import java.io.IOException;
import java.io.InputStream;
import java.io.Reader;
import java.io.StringReader;
import java.net.URL;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.Iterator;
import java.util.List;
import java.util.Map;

import org.openrdf.model.BNode;
import org.openrdf.model.Graph;
import org.openrdf.model.Resource;
import org.openrdf.model.Statement;
import org.openrdf.model.URI;
import org.openrdf.model.Value;
import org.openrdf.model.ValueFactory;
import org.openrdf.model.impl.GraphImpl;

import org.openrdf.rio.RdfDocumentWriter;
import org.openrdf.rio.n3.N3Writer;
import org.openrdf.rio.ntriples.NTriplesWriter;
import org.openrdf.rio.rdfxml.RdfXmlWriter;
import org.openrdf.rio.turtle.TurtleWriter;

import org.openrdf.sesame.admin.AdminListener;
import org.openrdf.sesame.admin.RdfAdmin;
import org.openrdf.sesame.admin.UpdateException;
import org.openrdf.sesame.config.AccessDeniedException;
import org.openrdf.sesame.config.UnknownRepositoryException;
import org.openrdf.sesame.constants.QueryLanguage;
import org.openrdf.sesame.constants.RDFFormat;
import org.openrdf.sesame.export.RdfExport;
import org.openrdf.sesame.query.GraphQuery;
import org.openrdf.sesame.query.GraphQueryResultListener;
import org.openrdf.sesame.query.MalformedQueryException;
import org.openrdf.sesame.query.QueryEvaluationException;
import org.openrdf.sesame.query.QueryResultsGraphBuilder;
import org.openrdf.sesame.query.QueryResultsTable;
import org.openrdf.sesame.query.QueryResultsTableBuilder;
import org.openrdf.sesame.query.TableQuery;
import org.openrdf.sesame.query.TableQueryResultListener;
import org.openrdf.sesame.query.rdql.RdqlEngine;
import org.openrdf.sesame.query.rql.RqlEngine;
import org.openrdf.sesame.query.serql.SerqlEngine;
import org.openrdf.sesame.repository.SesameRepository;
import org.openrdf.sesame.sail.RdfRepository;
import org.openrdf.sesame.sail.RdfSchemaSource;
import org.openrdf.sesame.sail.RdfSource;
import org.openrdf.sesame.sail.Sail;
import org.openrdf.sesame.sail.SailChangedEvent;
import org.openrdf.sesame.sail.SailChangedListener;
import org.openrdf.sesame.sail.SailUpdateException;
import org.openrdf.sesame.sail.StatementIterator;

/**
 * Entry point for access to a local Sesame repository. Create this object
 * using a LocalService.
 *
 * @see LocalService
 */
public class LocalRepository implements SesameRepository, SailChangedListener {

	/*----------+
	| Constants |
	+----------*/

	/**
	 * Constant defining default behavior of add methods: by default identical
	 * blank nodes will be 'joined'.
	 */
	private static final boolean JOIN_BLANKNODES_DEFAULT = true;
	
	/*---------------------+
	 | Variables            |
	 +---------------------*/

	private RdfSource _rdfSource;

	private String _id;

	private SerqlEngine _serqlQueryEngine;

	private RqlEngine _rqlQueryEngine;

	private RdqlEngine _rdqlQueryEngine;

	private RdfAdmin _rdfAdmin;

	private RdfExport _rdfExport;

	private LocalService _service;

	private List _listeners;

	/*---------------------+
	 | Constructors         |
	 +---------------------*/

	/**
	 * Creates a LocalRepository for the supplied Sail.
	 **/
	protected LocalRepository(String id, RdfSource rdfSource,
			LocalService service)
	{
		_id = id;
		_rdfSource = rdfSource;
		_service = service;

		_listeners = new ArrayList(0);

		if (_rdfSource instanceof RdfRepository) {
			((RdfRepository)_rdfSource).addListener(this);
		}
	}

	/*---------------------+
	 | Query methods        |
	 +---------------------*/

	/**
	 * Checks whether the current user has read access to this repository.
	 * @return <tt>true</tt> if the user has read access, <tt>false</tt> otherwise.
	 **/
	public boolean hasReadAccess() {
		try {
			return _service.hasReadAccess(_id);
		}
		catch (UnknownRepositoryException e) {
			throw new RuntimeException(e);
		}
	}

	/**
	 * Checks whether the current user has write access to this repository.
	 * @return <tt>true</tt> if the user has write access, <tt>false</tt> otherwise.
	 **/
	public boolean hasWriteAccess() {
		try {
			return _service.hasWriteAccess(_id);
		}
		catch (UnknownRepositoryException e) {
			throw new RuntimeException(e);
		}
	}

	/**
	 * Ensures that the current user has read access. If the user does not have
	 * read access, an <tt>AccessDeniedException</tt> will be thrown.
	 **/
	protected void _ensureReadAccess()
		throws AccessDeniedException
	{
		if (!hasReadAccess()) {
			throw new AccessDeniedException("Need read access");
		}
	}

	/**
	 * Ensures that the current user has write access. If the user does not have
	 * write access, an <tt>AccessDeniedException</tt> will be thrown.
	 **/
	protected void _ensureWriteAccess()
		throws AccessDeniedException
	{
		if (!hasWriteAccess()) {
			throw new AccessDeniedException("Need write access");
		}
	}

	// implements SesameRepository.performTableQuery(QueryLanguage, String)
	public QueryResultsTable performTableQuery(QueryLanguage language,
			String query)
		throws IOException, MalformedQueryException, QueryEvaluationException,
		AccessDeniedException
	{
		QueryResultsTableBuilder builder = new QueryResultsTableBuilder();

		performTableQuery(language, query, builder);

		return builder.getQueryResultsTable();
	}

	// implements SesameRepository.performTableQuery(QueryLanguage, String, TableQueryResultListener)
	public void performTableQuery(QueryLanguage language, String query,
			TableQueryResultListener listener)
		throws IOException, MalformedQueryException, QueryEvaluationException,
		AccessDeniedException
	{
		_ensureReadAccess();

		if (QueryLanguage.SERQL.equals(language)) {
			// SeRQL select query
			_performSeRQLSelectQuery(query, listener);
		}
		else if (QueryLanguage.RQL.equals(language)) {
			// RQL query
			_performRqlQuery(query, listener);
		}
		else if (QueryLanguage.RDQL.equals(language)) {
			// RDQL query
			_performRdqlQuery(query, listener);
		}
		else {
			throw new IllegalArgumentException("Unknown query language: "
					+ language);
		}
	}

	// implements SesameRepository.performGraphQuery(QueryLanguage, String, GraphQueryResultListener)
	public void performGraphQuery(QueryLanguage language, String query,
			GraphQueryResultListener listener)
		throws IOException, MalformedQueryException, QueryEvaluationException,
		AccessDeniedException
	{
		_ensureReadAccess();

		if (QueryLanguage.SERQL.equals(language)) {
			// SeRQL construct query
			_performSeRQLConstructQuery(query, listener);
		}
		else {
			// no other query language supports graph queries.
			throw new IllegalArgumentException("Query language "
					+ language.toString() + " does not support graph queries");
		}
	}

	public Graph performGraphQuery(QueryLanguage language, String query)
		throws IOException, MalformedQueryException, QueryEvaluationException,
		AccessDeniedException
	{
		_ensureReadAccess();

		QueryResultsGraphBuilder listener = new QueryResultsGraphBuilder();

		if (QueryLanguage.SERQL.equals(language)) {
			// SeRQL construct query
			_performSeRQLConstructQuery(query, listener);

			return listener.getGraph();
		}
		else {
			// no other query language supports graph queries.
			throw new IllegalArgumentException("Query language "
					+ language.toString() + " does not support graph queries");
		}
	}

	/**
	 * Performs an RQL-select query.
	 **/
	protected void _performRqlQuery(String query,
			TableQueryResultListener listener)
		throws IOException, MalformedQueryException, QueryEvaluationException
	{
		if (_rqlQueryEngine == null) {
			try {
				RdfSchemaSource rss = (RdfSchemaSource)_rdfSource;
				_rqlQueryEngine = new RqlEngine(rss);
			}
			catch (ClassCastException e) {
				throw new UnsupportedOperationException(
						"Repository is not an RdfSchemaSource");
			}
		}

		org.openrdf.sesame.query.rql.model.Query queryModel = _rqlQueryEngine.parseQuery(query);

		_rqlQueryEngine.evaluateQuery(queryModel, listener);
	}

	/**
	 * Performs an RDQL-select query.
	 **/
	protected void _performRdqlQuery(String query,
			TableQueryResultListener listener)
		throws IOException, MalformedQueryException, QueryEvaluationException
	{
		if (_rdqlQueryEngine == null) {
			_rdqlQueryEngine = new RdqlEngine(_rdfSource);
		}

		TableQuery tableQuery = _rdqlQueryEngine.parseQuery(query);

		_rdqlQueryEngine.evaluateQuery(tableQuery, listener);
	}

	/**
	 * Performs a SeRQL-select query.
	 **/
	protected void _performSeRQLSelectQuery(String query,
			TableQueryResultListener listener)
		throws IOException, MalformedQueryException, QueryEvaluationException
	{
		if (_serqlQueryEngine == null) {
			_serqlQueryEngine = new SerqlEngine(_rdfSource);
		}

		TableQuery tableQuery = _serqlQueryEngine.parseTableQuery(query);

		_serqlQueryEngine.evaluateSelectQuery(tableQuery, listener);
	}

	/**
	 * Performs a SeRQL-construct query.
	 **/
	protected void _performSeRQLConstructQuery(String query,
			GraphQueryResultListener listener)
		throws IOException, MalformedQueryException, QueryEvaluationException
	{
		if (_serqlQueryEngine == null) {
			_serqlQueryEngine = new SerqlEngine(_rdfSource);
		}

		GraphQuery graphQuery = _serqlQueryEngine.parseGraphQuery(query);

		_serqlQueryEngine.evaluateConstructQuery(graphQuery, listener);
	}

	/*---------------------+
	 | Update methods       |
	 +---------------------*/

	/**
	 * Creates an RdfAdmin object and assigns it to _rdfAdmin, if no _rdfAdmin
	 * object exists yet.
	 **/
	protected void _ensureRdfAdminCreated() {
		if (_rdfAdmin == null) {
			try {
				RdfRepository rdfRepository = (RdfRepository)_rdfSource;
				_rdfAdmin = new RdfAdmin(rdfRepository, _service);
			}
			catch (ClassCastException e) {
				throw new UnsupportedOperationException(
						"Repository is not an RdfRepository");
			}
		}
	}

	// implements SesameRepository.addData(URL, String, RDFFormat, boolean, AdminListener)
	public void addData(URL dataURL, String baseURI, RDFFormat format,
			boolean verifyData, AdminListener listener)
		throws IOException, AccessDeniedException
	{
		if (baseURI == null) {
			baseURI = dataURL.toExternalForm();
		}

		InputStream inputStream = dataURL.openStream();
		try {
			addData(inputStream, baseURI, format, verifyData, listener);
		}
		finally {
			inputStream.close();
		}
	}

	// implements SesameRepository.addData(File, String, RDFFormat, boolean, AdminListener)
	public void addData(File dataFile, String baseURI, RDFFormat format,
			boolean verifyData, AdminListener listener)
		throws FileNotFoundException, IOException, AccessDeniedException
	{
		InputStream inputStream = new FileInputStream(dataFile);
		try {
			addData(inputStream, baseURI, format, verifyData, listener);
		}
		finally {
			inputStream.close();
		}
	}

	// implements SesameRepository.addData(String, String, RDFFormat, boolean, AdminListener)
	public void addData(String data, String baseURI, RDFFormat format,
			boolean verifyData, AdminListener listener)
		throws IOException, AccessDeniedException
	{
		addData(new StringReader(data), baseURI, format, verifyData, listener);
	}

	// implements SesameRepository.addData(SesameRepository, AdminListener)
	public void addData(SesameRepository repository, AdminListener listener)
		throws IOException, AccessDeniedException
	{
		if (this == repository) {
			// Adding all data from this to this is useless. This special case
			// is handled to prevent potential deadlocks.
			return;
		}

		InputStream dataStream = repository.extractRDF(RDFFormat.RDFXML, true,
				true, true, false);
		try {
			addData(dataStream, "foo:bar", RDFFormat.RDFXML, false, listener);
		}
		finally {
			dataStream.close();
		}
	}

	// implements SesameRepository.addData(Reader, String, RDFFormat, boolean, AdminListener)
	public void addData(Reader reader, String baseURI, RDFFormat format,
			boolean verifyData, AdminListener listener)
		throws IOException, AccessDeniedException
	{
		_ensureWriteAccess();
		_ensureRdfAdminCreated();

		try {
			_rdfAdmin.addRdfModel(reader, baseURI, listener, format, verifyData);
		}
		catch (UpdateException e) {
			// ignore, error has been reported to the listener
		}
	}

	// implements SesameRepository.addData(InputStream, String, RDFFormat, boolean, AdminListener)
	public void addData(InputStream dataStream, String baseURI,
			RDFFormat format, boolean verifyData, AdminListener listener)
		throws IOException, AccessDeniedException
	{
		_ensureWriteAccess();
		_ensureRdfAdminCreated();

		try {
			_rdfAdmin.addRdfModel(dataStream, baseURI, listener, format, verifyData);
		}
		catch (UpdateException e) {
			// ignore, error has been reported to the listener
		}
	}

	// implements SesameRepository.extractData(RDFFormat, boolean, boolean, boolean, boolean)
	public InputStream extractRDF(RDFFormat format, boolean ontology,
			boolean instances, boolean explicitOnly, boolean niceOutput)
		throws IOException, AccessDeniedException
	{
		_ensureReadAccess();

		ByteArrayOutputStream baos = new ByteArrayOutputStream(8092);

		RdfDocumentWriter rdfDocWriter = null;
		if (RDFFormat.RDFXML.equals(format)) {
			rdfDocWriter = new RdfXmlWriter(baos);
		}
		else if (RDFFormat.NTRIPLES.equals(format)) {
			rdfDocWriter = new NTriplesWriter(baos);
		}
		else if (RDFFormat.N3.equals(format)) {
			rdfDocWriter = new N3Writer(baos);
		}
		else if (RDFFormat.TURTLE.equals(format)) {
			rdfDocWriter = new TurtleWriter(baos);
		}

		extractRDF(rdfDocWriter, ontology, instances, explicitOnly, niceOutput);

		return new ByteArrayInputStream(baos.toByteArray());
	}

	/**
	 * Extracts data from the repository and reports the triples to the supplied
	 * <tt>RdfDocumentWriter</tt>.
	 *
	 * @param rdfDocWriter The <tt>RdfDocumentWriter</tt> to report the triples to.
	 * @param ontology If <tt>true</tt> the ontological statements will be extracted.
	 * @param instances If <tt>true</tt> the instance (non-schema) statements will be extracted.
	 * @param explicitOnly If <tt>true</tt>, only the explicitly added statements will be extracted.
	 * @param niceOutput If <tt>true</tt>, the extracted statements will be sorted by their subject.
	 **/
	public void extractRDF(RdfDocumentWriter rdfDocWriter, boolean ontology,
			boolean instances, boolean explicitOnly, boolean niceOutput)
		throws IOException, AccessDeniedException
	{
		_ensureReadAccess();

		if (_rdfExport == null) {
			_rdfExport = new RdfExport();
		}

		if (_rdfSource instanceof RdfSchemaSource) {
			_rdfExport.exportRdf((RdfSchemaSource)_rdfSource, rdfDocWriter,
					ontology, instances, explicitOnly, niceOutput);
		}
		else {
			_rdfExport.exportRdf(_rdfSource, rdfDocWriter, niceOutput);
		}
	}

	// implements SesameRepository.removeStatements(Resource, URI, Value, AdminListener)
	public void removeStatements(Resource subject, URI predicate, Value object,
			AdminListener listener)
		throws IOException, AccessDeniedException
	{
		_ensureWriteAccess();
		_ensureRdfAdminCreated();

		try {
			_rdfAdmin.removeStatements(subject, predicate, object, listener);
		}
		catch (UpdateException e) {
			// ignore, error has been reported to the listener
		}
	}

	// implements SesameRepository.clear(AdminListener)
	public void clear(AdminListener listener)
		throws IOException, AccessDeniedException
	{
		_ensureWriteAccess();
		_ensureRdfAdminCreated();

		try {
			_rdfAdmin.clearRepository(listener);
		}
		catch (UpdateException e) {
			// ignore, error has been reported to the listener
		}
	}

	/**
	 * Gets the SAIL object of this repository. Note that no security
	 * restrictions are placed on the Sail; it is the developer's responsibility
	 * to check access restrictions before doing operations on the Sail.
	 */
	public Sail getSail() {
		return _rdfSource;
	}

	/**
	 * Shuts down the repository. Once shut down, the repository can no longer
	 * be queried or modified.
	 **/
	public synchronized void shutDown() {
		if (_rdfSource != null) {
			_rdfSource.shutDown();
			_rdfSource = null;

			_serqlQueryEngine = null;
			_rqlQueryEngine = null;
			_rdqlQueryEngine = null;
			_rdfAdmin = null;
			_rdfExport = null;
		}
	}

	//	 implements SesameRepository.mergeGraph(Graph)
	public void mergeGraph(Graph graph)
		throws IOException, AccessDeniedException
	{
		_ensureWriteAccess();

		// Write access given, therefore Sail is an RdfRepository.
		RdfRepository thisRep = (RdfRepository)_rdfSource;

		StatementIterator iter = graph.getStatements();

		try {
			thisRep.startTransaction();
			while (iter.hasNext()) {
				Statement st = iter.next();

				Resource subject = st.getSubject();
				URI predicate = st.getPredicate();
				Value object = st.getObject();
				thisRep.addStatement(subject, predicate, object);
			}
		}
		catch (SailUpdateException e) {
			// FIXME is this a correct conversion?
			throw new IOException(e.getMessage());
		}
		finally {
			thisRep.commitTransaction();
			iter.close();
		}
	}

	// implements SesameRepository.mergeGraph(QueryLanguage, String)
	public void mergeGraph(QueryLanguage language, String query)
		throws IOException, AccessDeniedException
	{
		try {
			Graph graph = performGraphQuery(language, query);
			mergeGraph(graph);
		}
		catch (QueryEvaluationException e) {
			throw new IOException(e.getMessage());
		}
		catch (MalformedQueryException e) {
			throw new IOException(e.getMessage());
		}
	}

	// implements SesameRepository.addGraph(Graph)
	public void addGraph(Graph graph) 
		throws IOException, AccessDeniedException
	{
		addGraph(graph, JOIN_BLANKNODES_DEFAULT);
	}
	
	public void addGraph(Graph graph, boolean joinBlankNodes)
		throws IOException, AccessDeniedException
	{

		_ensureWriteAccess();

		// Write access given, therefore Sail is an RdfRepository.
		RdfRepository thisRep = (RdfRepository)_rdfSource;
		
		Map bNodesMap = null;
		ValueFactory factory = null;
		
		if (!joinBlankNodes) {
			// addGraph needs to create new blank nodes to avoid accidental
			// merging of blank nodes in the repository.
			bNodesMap = new HashMap();
			factory = thisRep.getValueFactory();
		}

		StatementIterator iter = graph.getStatements();

		try {
			thisRep.startTransaction();
			while (iter.hasNext()) {
				Statement st = iter.next();

				Resource subject = st.getSubject();
				URI predicate = st.getPredicate();
				Value object = st.getObject();

				if (!joinBlankNodes) {
					if (subject instanceof BNode) {
						String bNodeId = ((BNode)subject).getID();
						if (bNodesMap.containsKey(bNodeId)) {
							// bnode was mapped before, reuse
							subject = (Resource)bNodesMap.get(bNodeId);
						}
						else {
							// create a new blank node and add it to the mapping.
							subject = factory.createBNode();
							bNodesMap.put(bNodeId, subject);
						}
					}
					
					if (object instanceof BNode) {
						String bNodeId = ((BNode)object).getID();
						if (bNodesMap.containsKey(bNodeId)) {
							// bnode was mapped before, reuse
							object = (Resource)bNodesMap.get(bNodeId);
						}
						else {
							// create a new blank node and add it to the mapping.
							object = factory.createBNode();
							bNodesMap.put(bNodeId, object);
						}
					}
				}

				thisRep.addStatement(subject, predicate, object);
			}
		}
		catch (SailUpdateException e) {
			// FIXME is this a correct conversion?
			throw new IOException(e.getMessage());
		}
		finally {
			thisRep.commitTransaction();
			iter.close();
		}
	}

	// implements SesameRepository.removeGraph(Graph)
	public void removeGraph(Graph graph)
		throws IOException, AccessDeniedException
	{
		_ensureWriteAccess();

		// Write access given, therefore Sail is an RdfRepository.
		RdfRepository thisRep = (RdfRepository)_rdfSource;

		StatementIterator iter = graph.getStatements();

		try {
			thisRep.startTransaction();
			while (iter.hasNext()) {
				Statement st = iter.next();
				thisRep.removeStatements(st.getSubject(), st.getPredicate(),
						st.getObject());
			}
		}
		catch (SailUpdateException e) {
			// FIXME is this a correct conversion?
			throw new IOException(e.getMessage());
		}
		finally {
			thisRep.commitTransaction();
			iter.close();
		}
	}

	/**
	 * Creates a Graph representation of this repository. Note that any changes
	 * on the Graph object will be reflected in this repository immediately!
	 * 
	 * @return a Graph object that uses this repository as the backing store.
	 * 
	 * @throws AccessDeniedException In case one does not have both read and
	 * write access on this repository.
	 */
	public Graph getGraph()
		throws AccessDeniedException
	{
		_ensureReadAccess();
		_ensureWriteAccess();
		return new GraphImpl(this);
	}

	// implements SesameRepository.addGraph(QueryLanguage, String)
	public void addGraph(QueryLanguage language, String query)
		throws IOException, AccessDeniedException
	{
		addGraph(language, query, JOIN_BLANKNODES_DEFAULT);
	}

	public void addGraph(QueryLanguage language, String query, boolean joinBlankNodes)
	throws IOException, AccessDeniedException
	{
		try {
			Graph graph = performGraphQuery(language, query);
			addGraph(graph, joinBlankNodes);
		}
		catch (QueryEvaluationException e) {
			throw new IOException(e.getMessage());
		}
		catch (MalformedQueryException e) {
			throw new IOException(e.getMessage());
		}
	}

	// implements SesameRepository.removeGraph(QueryLanguage, String)
	public void removeGraph(QueryLanguage language, String query)
		throws IOException, AccessDeniedException
	{
		try {
			Graph graph = performGraphQuery(language, query);
			removeGraph(graph);
		}
		catch (QueryEvaluationException e) {
			throw new IOException(e.getMessage());
		}
		catch (MalformedQueryException e) {
			throw new IOException(e.getMessage());
		}
	}

	public String getRepositoryId() {
		return _id;
	}

	public void addListener(LocalRepositoryChangedListener listener) {
		synchronized(_listeners) {
			_listeners.add(listener);
		} // end synchronized block
	}

	public void removeListener(LocalRepositoryChangedListener listener) {
		synchronized(_listeners) {
			_listeners.remove(listener);
		} // end synchronized block
	}

	// listener notifications

	private void _notifyRepositoryChange(LocalRepositoryChangedEvent event) {
		synchronized(_listeners) {
			Iterator listeners = _listeners.iterator();

			while (listeners.hasNext()) {
				LocalRepositoryChangedListener listener = (LocalRepositoryChangedListener)listeners.next();
				listener.repositoryChanged(event);
			}
		} // end synchronized block
	}

	/* (non-Javadoc)
	 * @see org.openrdf.sesame.sail.SailChangedListener#sailChanged(org.openrdf.sesame.sail.SailChangedEvent)
	 */
	public void sailChanged(SailChangedEvent event) {
		LocalRepositoryChangedEvent lrcEvent = new LocalRepositoryChangedEventImpl(
				event.statementsAdded(), event.statementsRemoved());
		_notifyRepositoryChange(lrcEvent);
	}
}
