/*  Sesame - Storage and Querying architecture for RDF and RDF Schema
 *  Copyright (C) 2001-2006 Aduna
 *
 *  Contact: 
 *  	Aduna
 *  	Prinses Julianaplein 14 b
 *  	3817 CS Amersfoort
 *  	The Netherlands
 *  	tel. +33 (0)33 465 99 87
 *  	fax. +33 (0)33 465 99 87
 *
 *  	http://aduna-software.com/
 *  	http://www.openrdf.org/
 *  
 *  This library is free software; you can redistribute it and/or
 *  modify it under the terms of the GNU Lesser General Public
 *  License as published by the Free Software Foundation; either
 *  version 2.1 of the License, or (at your option) any later version.
 *
 *  This library is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 *  Lesser General Public License for more details.
 *
 *  You should have received a copy of the GNU Lesser General Public
 *  License along with this library; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

package org.openrdf.sesame.query.serql.parser;

import java.util.List;

import org.openrdf.sesame.sail.query.TriplePattern;
import org.openrdf.sesame.sail.query.Var;

/**
 * Represents a reified statement. A Reified_stat is equal to a Path_expr, but without a tail.
 * This means the right node of a Reified_stat is equal to a Path_expr without edges and right
 * nodes. A Reified_stat has an extra instance variable, i.e. its identifier. A Reified_stat
 * {S} P {O} with identifier X expands to X rdf:type rdf:Statement, X rdf:subject S, X
 * rdf:predicate P, X rdf:object O and S P O.
 *
 * @author Peter van 't Hof
 * @author Arjohn Kampman
 * @version $Revision: 1.5.4.2 $
 */
public class ReifiedPathElement extends PathElement {
	
/*----------+
| Variables |
+----------*/

	/**
	 * Identifier of this Reified_stat.
	 **/
	Var _id;

/*------------+
| Constructor |
+------------*/

	/**
	 * Creates a new ReifiedPathElement.
	 *
	 * @param id Id of this reified statement.
	 * @param leftNode Left node of this reified statement.
	 * @param edge Edge of this reified statement.
	 * @param rightNode Right node of this reified statement.
	 */
	public ReifiedPathElement(Var id, List leftNode, Var edge, List rightNode) {
		super(leftNode, edge, rightNode);
		_id = id;
	}

/*--------+
| Methods |
+--------*/

	/**
	 * Gets the identifier of this reified statement.
	 *
	 * @return Id of this Reified_stat.
	 */
	public Var getId() {
		return _id;
	}

	// Overrides PathElement.getTriplePatterns()
	public List getTriplePatterns() {
		List triplePatterns = super.getTriplePatterns();

		// _id rdf:type rdf:Statement.
		triplePatterns.add(new TriplePattern(
				_id, SerqlParser.RDF_TYPE_VAR, SerqlParser.RDF_STATEMENT_VAR));
		
		return triplePatterns;
	}

	// Overrides PathElement._createTriplePatterns.
	protected void _createTriplePatterns(
		Var subject, Var predicate, Var object, List triplePatterns)
	{
		// _id rdf:subject subject.
		triplePatterns.add(new TriplePattern(
				_id, SerqlParser.RDF_SUBJECT_VAR, subject));

		// _id rdf:predicate predicate.
		triplePatterns.add(new TriplePattern(
				_id, SerqlParser.RDF_PREDICATE_VAR, predicate));

		// _id rdf:object object.
		triplePatterns.add(new TriplePattern(
				_id, SerqlParser.RDF_OBJECT_VAR, object));
	}
}
