/*  Sesame - Storage and Querying architecture for RDF and RDF Schema
 *  Copyright (C) 2001-2006 Aduna
 *
 *  Contact: 
 *  	Aduna
 *  	Prinses Julianaplein 14 b
 *  	3817 CS Amersfoort
 *  	The Netherlands
 *  	tel. +33 (0)33 465 99 87
 *  	fax. +33 (0)33 465 99 87
 *
 *  	http://aduna-software.com/
 *  	http://www.openrdf.org/
 *  
 *  This library is free software; you can redistribute it and/or
 *  modify it under the terms of the GNU Lesser General Public
 *  License as published by the Free Software Foundation; either
 *  version 2.1 of the License, or (at your option) any later version.
 *
 *  This library is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 *  Lesser General Public License for more details.
 *
 *  You should have received a copy of the GNU Lesser General Public
 *  License along with this library; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

package org.openrdf.sesame.query.serql.parser;

import java.util.ArrayList;
import java.util.Iterator;
import java.util.List;

import org.openrdf.model.Value;

import org.openrdf.sesame.sail.query.DirectSubClassOf;
import org.openrdf.sesame.sail.query.DirectSubPropertyOf;
import org.openrdf.sesame.sail.query.DirectType;
import org.openrdf.sesame.sail.query.PathExpression;
import org.openrdf.sesame.sail.query.TriplePattern;
import org.openrdf.sesame.sail.query.Var;

/**
 */
public class PathElement {

/*----------+
| Variables |
+----------*/

	/**
	 * Left node of this PathElement. Contains objects of type Var.
	 **/
	protected List _subjectVars;

	/**
	 * Edge of this PathElement.
	 **/
	protected Var _edge;

	/**
	 * Right node of this PathElement. Contains objects of type Var.
	 **/
	protected List _objectVars;

/*-------------+
| Constructors |
+-------------*/

	/**
	 * Creates a new PathElement.
	 *
	 * @param subjectVars Left node of this PathElement.
	 * @param edge Edge of this PathElement.
	 * @param objectVars Right node of this PathElement.
	 */
	public PathElement(List subjectVars, Var edge, List objectVars) {
		_subjectVars = subjectVars;
		_edge = edge;
		_objectVars = objectVars;
	}

/*--------+
| Methods |
+--------*/

	public List getSubjectVars() {
		return _subjectVars;
	}

	public Var getEdge() {
		return _edge;
	}

	public List getObjectVars() {
		return _objectVars;
	}

	/**
	 * Creates a list of triple patterns that can represent this path element.
	 *
	 * @return A List of TriplePattern objects.
	 */
	public List getTriplePatterns() {
		List triplePatterns = new ArrayList();

		Iterator leftIter = _subjectVars.iterator();
		while (leftIter.hasNext()) {
			Var subject = (Var)leftIter.next();
			
			Iterator rightIter = _objectVars.iterator();
			while (rightIter.hasNext()) {
				Var object = (Var)rightIter.next();
				
				_createTriplePatterns(subject, _edge, object, triplePatterns);
			}
		}

		return triplePatterns;
	}
	
	/**
	 * Creates a TriplePattern with the supplied subject, predicate and object
	 * and adds it to the supplied list. This method can be overridden by
	 * subclasses if more TriplePatterns need to be created, e.g. by
	 * <tt>ReifiedPathElement</tt>.
	 */
	protected void _createTriplePatterns(
		Var subject, Var predicate, Var object, List triplePatterns)
	{
		PathExpression newPE;

		Value predValue = predicate.getValue();

		if (SerqlParser.SERQL_DIRECTSUBCLASSOF.equals(predValue)) {
			newPE = new DirectSubClassOf(subject, object);
		}
		else if (SerqlParser.SERQL_DIRECTSUBPROPERTYOF.equals(predValue)) {
			newPE = new DirectSubPropertyOf(subject, object);
		}
		else if (SerqlParser.SERQL_DIRECTTYPE.equals(predValue)) {
			newPE = new DirectType(subject, object);
		}
		else {
			newPE = new TriplePattern(subject, predicate, object);
		}

		triplePatterns.add(newPE);
	}
	
	/**
	 * Returns String representation of this Path_expr.
	 *
	 * @return String representation of this Path_expr.
	 */
	public String toString() {
		String result = "{";

		Iterator i = _subjectVars.iterator();
		while (i.hasNext()) {
			result += i.next().toString();

			if (i.hasNext()) {
				result += ", ";
			}				
		}

		result += "} ";
		result += _edge.toString();

		result += " {";
		i = _objectVars.iterator();
		while (i.hasNext()) {
			result += i.next().toString();

			if (i.hasNext()) {
				result += ", ";
			}				
		}
		result += "}";

		return result;
	}
}
