/*  Sesame - Storage and Querying architecture for RDF and RDF Schema
 *  Copyright (C) 2001-2006 Aduna
 *
 *  Contact: 
 *  	Aduna
 *  	Prinses Julianaplein 14 b
 *  	3817 CS Amersfoort
 *  	The Netherlands
 *  	tel. +33 (0)33 465 99 87
 *  	fax. +33 (0)33 465 99 87
 *
 *  	http://aduna-software.com/
 *  	http://www.openrdf.org/
 *  
 *  This library is free software; you can redistribute it and/or
 *  modify it under the terms of the GNU Lesser General Public
 *  License as published by the Free Software Foundation; either
 *  version 2.1 of the License, or (at your option) any later version.
 *
 *  This library is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 *  Lesser General Public License for more details.
 *
 *  You should have received a copy of the GNU Lesser General Public
 *  License along with this library; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

package org.openrdf.sesame.query.serql;

import java.io.IOException;
import java.io.StringReader;

import org.openrdf.util.StringUtil;

import org.openrdf.sesame.query.GraphQuery;
import org.openrdf.sesame.query.GraphQueryResultListener;
import org.openrdf.sesame.query.MalformedQueryException;
import org.openrdf.sesame.query.QueryEvaluationException;
import org.openrdf.sesame.query.TableQuery;
import org.openrdf.sesame.query.TableQueryResultListener;
import org.openrdf.sesame.query.serql.parser.ParseException;
import org.openrdf.sesame.query.serql.parser.SerqlParser;
import org.openrdf.sesame.query.serql.parser.TokenMgrError;
import org.openrdf.sesame.sail.RdfSource;

/**
 * QueryEngine for SeRQL queries.
 **/
public class SerqlEngine {

/*---------------+
| Variables      |
+---------------*/

	/**
	 * RDF (read-only) repository
	 **/
	protected RdfSource _rdfSource;

/*---------------+
| Constructors   |
+---------------*/

	/**
	 * Creates a new QueryEngine that will use the supplied RdfSource
	 * for retrieving the data.
	 *
	 * @exception IllegalArgumentException If the supplied RdfSource is
	 * equal to null.
	 **/
	public SerqlEngine(RdfSource rdfSource) {
		if (rdfSource == null) {
			throw new IllegalArgumentException("rdfSource must not be nul");
		}
		_rdfSource = rdfSource;
	}

/*---------------+
| Methods        |
+---------------*/

	/**
	 * Evaluates the supplied SerqlQuery and reports the results to the
	 * supplied QueryResultListener.
	 *
	 * @exception IOException If an I/O error occurred during the writing
	 * of the results.
	 * @exception QueryEvaluationException If an error occurred during query
	 * evaluation.
	 **/
	public void evaluateSelectQuery(TableQuery tableQuery, TableQueryResultListener tableListener)
		throws IOException, QueryEvaluationException
	{   
		tableQuery.optimize(_rdfSource);
		tableQuery.evaluate(_rdfSource, tableListener);
	}

	public void evaluateConstructQuery(GraphQuery graphQuery, GraphQueryResultListener graphListener)
		throws IOException, QueryEvaluationException
	{
		graphQuery.optimize(_rdfSource);
		graphQuery.evaluate(_rdfSource, graphListener);
	}

	/**
	 * Parses the supplied SeRQL-select query and builds a query model from it.
	 * @return The query model.
	 *
	 * @exception MalformedQueryException If the query was syntactically
	 * incorrect.
	 **/
	public TableQuery parseTableQuery(String tableQuery)
		throws MalformedQueryException
	{
		tableQuery = _swapUsingNamespace(tableQuery);

		SerqlParser serqlParser = new SerqlParser(new StringReader(tableQuery) );

		try {
			return serqlParser.parseTableQuery();
		}
		catch (TokenMgrError e) {
			throw new MalformedQueryException(e);
		}
		catch (ParseException e) {
			throw new MalformedQueryException(e);
		}
	}

	/**
	 * Parses the supplied SeRQL-construct query and builds a query model from it.
	 * @return The query model.
	 *
	 * @exception MalformedQueryException If the query was syntactically
	 * incorrect.
	 **/
	public GraphQuery parseGraphQuery(String graphQuery)
		throws MalformedQueryException
	{
		graphQuery = _swapUsingNamespace(graphQuery);

		SerqlParser serqlParser = new SerqlParser(new StringReader(graphQuery) );

		try {
			return serqlParser.parseGraphQuery();
		}
		catch (TokenMgrError e) {
			throw new MalformedQueryException(e);
		}
		catch (ParseException e) {
			throw new MalformedQueryException(e);
		}
	}

	/**
	 * The USING NAMESPACE clause is the last clause in a query,
	 * which is quite inconvenient for the query parser. This method
	 * remove this clause from the end of the query and inserts it
	 * at the start of the query.
	 **/
	private String _swapUsingNamespace(String query) {
		// convert query to lower case
		String queryLC = query.toLowerCase();

		// search for last occurence of "using"
		int usingIdx = StringUtil.lastIndexOfWord(queryLC, "using");

		String result;

		if (usingIdx != -1) {
			// Swap the part before and after the 'using' keyword
			String part1 = query.substring(0, usingIdx);
			String part2 = query.substring(usingIdx);

			result = part2 + "\n" + part1;
		}
		else {
			// No 'using' clause
			result = query;
		}

		return result;
	}
}
