/*  Sesame - Storage and Querying architecture for RDF and RDF Schema
 *  Copyright (C) 2001-2006 Aduna
 *
 *  Contact: 
 *  	Aduna
 *  	Prinses Julianaplein 14 b
 *  	3817 CS Amersfoort
 *  	The Netherlands
 *  	tel. +33 (0)33 465 99 87
 *  	fax. +33 (0)33 465 99 87
 *
 *  	http://aduna-software.com/
 *  	http://www.openrdf.org/
 *  
 *  This library is free software; you can redistribute it and/or
 *  modify it under the terms of the GNU Lesser General Public
 *  License as published by the Free Software Foundation; either
 *  version 2.1 of the License, or (at your option) any later version.
 *
 *  This library is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 *  Lesser General Public License for more details.
 *
 *  You should have received a copy of the GNU Lesser General Public
 *  License along with this library; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

package org.openrdf.sesame.query.rql.model;

import java.io.IOException;
import java.util.HashSet;

import org.openrdf.model.Value;

import org.openrdf.sesame.query.QueryEvaluationException;
import org.openrdf.sesame.query.TableQueryResultListener;
import org.openrdf.sesame.sail.RdfSchemaSource;
import org.openrdf.sesame.sail.ValueIterator;

/** 
 * Union represents a set query where the result is the union of the
 * result sets of the two arguments. The arguments must be
 * select-from-where queries which have the same number of items in their
 * select clause.
 *
 * Query syntax:
 * <code>
 * ( select .... from .... [where ....]? )
 * union
 * ( select .... from .... [where ....]? )
 * </code>
 *
 * @author Jeen Broekstra
 * @version $Revision: 1.7.2.2 $
 */
public class Union implements SetQuery {

	protected SfwQuery _arg1;

	protected SfwQuery _arg2;

	protected HashSet resultSet;

	protected int _projectionSize;

	public Union(SfwQuery query1, SfwQuery query2) {
		_arg1 = query1;
		_arg2 = query2;
		resultSet = new HashSet();
	}

	public void evaluate(RdfSchemaSource rss, TableQueryResultListener listener)
		throws QueryEvaluationException
	{
		int proj1Size = _arg1.getProjection().size();
		int proj2Size = _arg2.getProjection().size();
		if (proj1Size != proj2Size) {
			throw new QueryEvaluationException("projection sizes must be equal");
		}
		else {
			_projectionSize = proj1Size;
			String columnHeaders[] = new String[_projectionSize];
			Query projectionQ;

			for (int i = 0; i < _projectionSize; i++) {
				projectionQ = (Query)_arg1.getProjection().get(i);
				if (projectionQ instanceof DataVar) {
					if (((DataVar)projectionQ).getName().equals("NULL")) {
						projectionQ = (Query)_arg2.getProjection().get(i);
					}
				}
				columnHeaders[i] = projectionQ.getQuery();
			}

			UnionIterator unionIter = new UnionIterator(rss, _arg1, _arg2);
			try {
				Value val;
				Projection projection;
				listener.startTableQueryResult(columnHeaders);

				while (unionIter.hasNext()) {
					listener.startTuple();
					projection = unionIter.next();
					for (int i = 0; i < _projectionSize; i++) {
						val = projection.get(i);
						listener.tupleValue(val);
					}
					listener.endTuple();
				}
				listener.endTableQueryResult();
			}
			catch (IOException e) {
				unionIter.close();
				throw new QueryEvaluationException(e);
			}

		}
	}

	public ValueIterator getResources(RdfSchemaSource rss) {
		// FIXME we need a datastructure for projections that fits in a
		// ValueIterator.
		//return new UnionIterator(rss);
		return null;
	}

	public boolean returnsSet() {
		return true;
	}

	public SfwQuery getArg1() {
		return _arg1;
	}

	public SfwQuery getArg2() {
		return _arg2;
	}

	public String getQuery() {
		return this.toString();
	}

	public String toString() {
		StringBuffer result = new StringBuffer();

		result.append("( ");
		result.append(_arg1.getQuery());
		result.append(" )");
		result.append("\nunion\n");
		result.append("( ");
		result.append(_arg2.getQuery());
		result.append(" )\n");
		return result.toString();
	}

	static class UnionIterator {

		protected boolean _hasNext;

		protected ValueIterator _arg1Iterator;

		protected ValueIterator _arg2Iterator;

		private Projection _nextValue;

		private HashSet _resultSet;

		private boolean _initialized;

		public UnionIterator(RdfSchemaSource rss, SfwQuery arg1, SfwQuery arg2)
			throws QueryEvaluationException
		{

			_arg1Iterator = arg1.getResources(rss);
			_arg2Iterator = arg2.getResources(rss);

			_resultSet = new HashSet();
		}

		public boolean hasNext()
			throws QueryEvaluationException
		{
			if (_nextValue == null && !_initialized) {
				_nextValue = _getNextValue();
				_hasNext = (_nextValue != null);
			}
			return _hasNext;
		}

		public Projection next()
			throws QueryEvaluationException
		{
			Projection result = null;
			if (this.hasNext()) {
				result = _nextValue;
				_nextValue = _getNextValue();
				_hasNext = (_nextValue != null);

				return result;
			}
			throw new java.util.NoSuchElementException();
		}

		public void close() {
			if (!_initialized) {
				_arg1Iterator.close();
			}
			_arg2Iterator.close();
		}

		protected void finalize() {
			close();
		}

		/*
		 * Iterates over both arguments and returns the next union value.
		 * Return null if no next value can be found.
		 */
		private Projection _getNextValue()
			throws QueryEvaluationException
		{
			Projection result = null;

			if (_arg1Iterator.hasNext()) {
				result = (Projection)_arg1Iterator.next();
				_resultSet.add(result);
			}
			else {
				if (!_initialized) {
					_arg1Iterator.close();
					_initialized = true;
				}
				Projection projection;
				while (_arg2Iterator.hasNext()) {
					projection = (Projection)_arg2Iterator.next();
					if (!_resultSet.contains(projection)) {
						result = projection;
						break;
					}
				}
			}
			return result;
		}
	} // end inner class

}
