/*  Sesame - Storage and Querying architecture for RDF and RDF Schema
 *  Copyright (C) 2001-2006 Aduna
 *
 *  Contact: 
 *  	Aduna
 *  	Prinses Julianaplein 14 b
 *  	3817 CS Amersfoort
 *  	The Netherlands
 *  	tel. +33 (0)33 465 99 87
 *  	fax. +33 (0)33 465 99 87
 *
 *  	http://aduna-software.com/
 *  	http://www.openrdf.org/
 *  
 *  This library is free software; you can redistribute it and/or
 *  modify it under the terms of the GNU Lesser General Public
 *  License as published by the Free Software Foundation; either
 *  version 2.1 of the License, or (at your option) any later version.
 *
 *  This library is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 *  Lesser General Public License for more details.
 *
 *  You should have received a copy of the GNU Lesser General Public
 *  License along with this library; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

package org.openrdf.sesame.query.rql.model;

import java.io.IOException;

import org.openrdf.model.Statement;
import org.openrdf.model.Value;
import org.openrdf.model.impl.URIImpl;

import org.openrdf.sesame.query.QueryEvaluationException;
import org.openrdf.sesame.query.TableQueryResultListener;
import org.openrdf.sesame.query.rql.model.iterators.StatementSubjectIterator;
import org.openrdf.sesame.sail.RdfSchemaSource;
import org.openrdf.sesame.sail.ResourceIterator;
import org.openrdf.sesame.sail.StatementIterator;
import org.openrdf.sesame.sail.ValueIterator;
import org.openrdf.sesame.sail.util.SingleResourceIterator;

public class URI implements PropertyQuery, ClassQuery, DataQuery {

	protected org.openrdf.model.URI _uri;

	public URI(String uri) {
		_uri = new URIImpl(uri);
	}

	public URI(org.openrdf.model.URI uri) {
		_uri = uri;
	}

	public String getQuery() {
		return _uri.getLocalName();
	}

	public void evaluate(RdfSchemaSource rss, TableQueryResultListener listener)
		throws QueryEvaluationException
	{
		try {
			// URI as top-level query
			if (rss.isClass(_uri)) {
				String columnHeaders[] = new String[1];
				columnHeaders[0] = this.getQuery();
				listener.startTableQueryResult(columnHeaders);

				StatementIterator iter = rss.getType(null, _uri);

				try {
					while (iter.hasNext()) {
						listener.startTuple();
						listener.tupleValue(iter.next().getSubject());
						listener.endTuple();
					}
				}
				finally {
					iter.close();
				}
			}
			else if (rss.isProperty(_uri)) {
				String columnHeaders[] = new String[2];
				columnHeaders[0] = "source of \"" + this.getQuery() + "\"";
				columnHeaders[1] = "target of \"" + this.getQuery() + "\"";
				listener.startTableQueryResult(columnHeaders);

				StatementIterator iter = rss.getStatements(null, _uri, null);

				try {
					while (iter.hasNext()) {
						Statement st = iter.next();

						listener.startTuple();
						listener.tupleValue(st.getSubject());
						listener.tupleValue(st.getObject());
						listener.endTuple();
					}
				}
				finally {
					iter.close();
				}
			}
			else {
				String columnHeaders[] = new String[1];
				columnHeaders[0] = this.getQuery();
				listener.startTableQueryResult(columnHeaders);

				listener.startTuple();
				listener.tupleValue(_uri);
				listener.endTuple();
			}

			listener.endTableQueryResult();
		}
		catch (IOException e) {
			throw new QueryEvaluationException(e);
		}
	}

	public ValueIterator getResources(RdfSchemaSource rss) {
		return new SingleResourceIterator(_uri);
	}

	public ResourceIterator getClasses(RdfSchemaSource rss) {
		// Assume the URI is a class URI
		return new SingleResourceIterator(_uri);
	}

	public ResourceIterator getProperties(RdfSchemaSource rss) {
		// Assume the URI is a property URI
		return new SingleResourceIterator(_uri);
	}

	public ValueIterator getInstances(RdfSchemaSource rss) {
		if (rss.isClass(_uri)) {
			StatementIterator statIter = rss.getType(null, _uri);
			return new StatementSubjectIterator(statIter);
		}
		else {
			return new SingleResourceIterator(_uri);
		}
	}

	public boolean returnsSet() {
		return false;
	}

	public Value getValue() {
		return _uri;
	}

	public String toString() {
		return _uri.getURI();
	}
}
