/*  Sesame - Storage and Querying architecture for RDF and RDF Schema
 *  Copyright (C) 2001-2006 Aduna
 *
 *  Contact: 
 *  	Aduna
 *  	Prinses Julianaplein 14 b
 *  	3817 CS Amersfoort
 *  	The Netherlands
 *  	tel. +33 (0)33 465 99 87
 *  	fax. +33 (0)33 465 99 87
 *
 *  	http://aduna-software.com/
 *  	http://www.openrdf.org/
 *  
 *  This library is free software; you can redistribute it and/or
 *  modify it under the terms of the GNU Lesser General Public
 *  License as published by the Free Software Foundation; either
 *  version 2.1 of the License, or (at your option) any later version.
 *
 *  This library is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 *  Lesser General Public License for more details.
 *
 *  You should have received a copy of the GNU Lesser General Public
 *  License along with this library; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

package org.openrdf.sesame.query.rql.model;

import java.io.IOException;
import java.util.List;

import org.openrdf.model.Resource;
import org.openrdf.model.Value;

import org.openrdf.sesame.query.QueryEvaluationException;
import org.openrdf.sesame.query.TableQueryResultListener;
import org.openrdf.sesame.sail.RdfSchemaSource;
import org.openrdf.sesame.sail.ResourceIterator;
import org.openrdf.sesame.sail.ValueIterator;

public class SfwQuery implements ResourceQuery {

	protected List _select;

	protected List _from;

	protected BooleanQuery _where;

	public SfwQuery(List select, List from, BooleanQuery where) {
		_select = select;
		_from = from;
		_where = where;
	}

	public void evaluate(RdfSchemaSource rss, TableQueryResultListener listener)
		throws QueryEvaluationException
	{
		try {
			int size = _select.size();
			Query projection = null;
			String columnHeaders[] = new String[size];

			for (int i = 0; i < size; i++) {
				projection = (Query)_select.get(i);
				columnHeaders[i] = projection.getQuery();
			}

			listener.startTableQueryResult(columnHeaders);

			boolean instantiated = _findNext(rss, true);

			while (instantiated) {
				if (_where == null || _where.isTrue(rss)) {
					_exportResults(rss, listener);
				}
				instantiated = _findNext(rss, false);
			}

			listener.endTableQueryResult();
		}
		catch (IOException e) {
			// Clear all selectors
			for (int i = _from.size() - 1; i >= 0; i--) {
				Selector selector = (Selector)_from.get(i);
				selector.clear();
			}

			throw new QueryEvaluationException(e);
		}
	}

	public ValueIterator getResources(RdfSchemaSource rss)
		throws QueryEvaluationException
	{
		// FIXME does not apply to set operations really.
		/*
		 if (_select.size() != 1) {
		 throw new QueryEvaluationException(
		 "More than one variable in select clause.");
		 }
		 */
		return new SfwIterator(rss);
	}

	/**
	 * Finds the next result. If 'firstCall' is true, then this method will
	 * start with the initialization of the first selector. Otherwise, this
	 * method will assume that all selectors have been initialization before
	 * and will start/continue with the last selector.
	 *
	 * @return true if a next results has been found, false otherwise.
	 * @throws QueryEvaluationException
	 **/
	private boolean _findNext(RdfSchemaSource rss, boolean firstCall)
		throws QueryEvaluationException
	{
		int lastIndex = _from.size() - 1;
		int index;
		Selector selector;

		if (firstCall) {
			index = 0;
			selector = (Selector)_from.get(index);
			selector.initialize(rss);
		}
		else {
			index = lastIndex;
			selector = (Selector)_from.get(index);
		}

		while (true) {
			if (selector.selectNext(rss)) {
				// Selector instantiated

				if (index == lastIndex) {
					// Last selector instantiated
					return true;
				}
				else {
					// Instantiate next selector
					index++;
					selector = (Selector)_from.get(index);
					selector.initialize(rss);
				}
			}
			else {
				// No more results found, go back in recursion
				selector.clear();

				if (index == 0) {
					// No more results
					return false;
				}
				else {
					index--;
					selector = (Selector)_from.get(index);
				}
			}
		}
	}

	private void _exportResults(RdfSchemaSource rss,
			TableQueryResultListener listener)
		throws QueryEvaluationException
	{
		try {
			listener.startTuple();

			int selectSize = _select.size();
			for (int i = 0; i < selectSize; i++) {
				ResourceQuery q = (ResourceQuery)_select.get(i);

				if (q instanceof Var) {
					Var var = (Var)q;
					Value varValue = var.getValue();
					listener.tupleValue(varValue);
				}
				else if (q instanceof URI) {
					URI uri = (URI)q;
					listener.tupleValue(uri.getValue());
				}
				else if (q instanceof Domain) {
					Domain domain = (Domain)q;
					ResourceIterator iter = domain.getClasses(rss);
					try {
						listener.tupleValue(iter.next());
					}
					finally {
						iter.close();
					}
				}
				else if (q instanceof Range) {
					Range range = (Range)q;
					ResourceIterator iter = range.getClasses(rss);
					try {
						listener.tupleValue(iter.next());
					}
					finally {
						iter.close();
					}
				}
				else if (q.returnsSet()) {
					ResourceIterator iter = (ResourceIterator)q.getResources(rss);

					// FIXME creating an ad-hoc Intersection object is not
					// very clean, conceptually. We need a more general
					// set object.
					Intersection qResources = new Intersection();

					while (iter.hasNext()) {
						qResources.add((Resource)iter.next());
					}
					try {
						listener.tupleValue(qResources);
					}
					finally {
						iter.close();
					}
				}
			}

			listener.endTuple();
		}
		catch (IOException e) {
			throw new QueryEvaluationException(e);
		}
	}

	public List getFromPart() {
		return _from;
	}

	public void setFromPart(List from) {
		_from = from;
	}

	public BooleanQuery getWherePart() {
		return _where;
	}

	public void setWherePart(BooleanQuery newWhere) {
		_where = newWhere;
	}

	public List getProjection() {
		return _select;
	}

	public boolean returnsSet() {
		return true;
	}

	public String getQuery() {
		return this.toString();
	}

	public String toString() {
		StringBuffer result = new StringBuffer();

		result.append("select ");
		int selectSize = _select.size();
		if (selectSize == 0) {
			result.append("*");
		}
		else {
			for (int i = 0; i < selectSize; i++) {
				ResourceQuery q = (ResourceQuery)_select.get(i);
				result.append(q.toString());
				if (i < selectSize - 1) {
					result.append(", ");
				}
			}
		}

		result.append("\nfrom\n");
		int nofSelectors = _from.size();
		for (int i = 0; i < nofSelectors; i++) {
			Selector selector = (Selector)_from.get(i);
			result.append("\t");
			result.append(selector.toString());
			if (i < nofSelectors - 1) {
				result.append(", ");
			}
			result.append("\n");
		}

		if (_where != null) {
			result.append("where ");
			result.append(_where.toString());
		}

		return result.toString();
	}

	/*
	 Internal class that can iterate over SfwQueries, returning one
	 value at a time, per row.
	 */
	class SfwIterator implements ValueIterator {

		protected RdfSchemaSource _rss;

		protected boolean _hasNext;

		protected ResourceQuery[] _projQuery;

		protected int _projSize, _projCounter;

		public SfwIterator(RdfSchemaSource rss)
			throws QueryEvaluationException
		{
			_rss = rss;
			_projCounter = 0;
			_hasNext = false;
			_projSize = _select.size();
			_projQuery = new ResourceQuery[_projSize];

			for (int i = 0; i < _projSize; i++) {
				_projQuery[i] = (ResourceQuery)_select.get(i);
			}
			boolean resultFound = _findNext(rss, true);

			while (resultFound) {
				if (_where == null || _where.isTrue(rss)) {
					_hasNext = true;
					break;
				}
				resultFound = _findNext(rss, false);
			}
		}

		public boolean hasNext() {
			return _hasNext;
		}

		public Value next()
			throws QueryEvaluationException
		{
			if (_hasNext) {
				ValueIterator valIter;
				Value result = new Projection(_select.size());

				for (int i = 0; i < _select.size(); i++) {
					valIter = _projQuery[i].getResources(_rss);
					if (valIter.hasNext()) {
						((Projection)result).add(valIter.next());
					}
					else { // null value in select
						((Projection)result).add(null);
					}
				}

				boolean resultFound = _findNext(_rss, false);
				_hasNext = false;

				while (resultFound) {
					if (_where == null || _where.isTrue(_rss)) {
						_hasNext = true;
						break;
					}
					resultFound = _findNext(_rss, false);
				}
				return result;
			}

			throw new java.util.NoSuchElementException();
		}

		public void close() {
			if (_hasNext) {
				// Do this only once:
				_hasNext = false;

				// Clear any remaining results
				for (int i = _from.size() - 1; i >= 0; i--) {
					Selector selector = (Selector)_from.get(i);
					selector.clear();
				}
			}
		}

		protected void finalize() {
			close();
		}
	}
}
