/*  Sesame - Storage and Querying architecture for RDF and RDF Schema
 *  Copyright (C) 2001-2006 Aduna
 *
 *  Contact: 
 *  	Aduna
 *  	Prinses Julianaplein 14 b
 *  	3817 CS Amersfoort
 *  	The Netherlands
 *  	tel. +33 (0)33 465 99 87
 *  	fax. +33 (0)33 465 99 87
 *
 *  	http://aduna-software.com/
 *  	http://www.openrdf.org/
 *  
 *  This library is free software; you can redistribute it and/or
 *  modify it under the terms of the GNU Lesser General Public
 *  License as published by the Free Software Foundation; either
 *  version 2.1 of the License, or (at your option) any later version.
 *
 *  This library is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 *  Lesser General Public License for more details.
 *
 *  You should have received a copy of the GNU Lesser General Public
 *  License along with this library; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

package org.openrdf.sesame.query.rql.model;

import java.util.Iterator;

import org.openrdf.model.Resource;
import org.openrdf.model.Value;

import org.openrdf.sesame.query.QueryEvaluationException;
import org.openrdf.sesame.query.rql.model.iterators.StatementSubjectIterator;
import org.openrdf.sesame.sail.RdfSchemaSource;
import org.openrdf.sesame.sail.ResourceIterator;

public class RangeSelector implements Selector {

	protected ClassQuery _rangeQ;
	protected PropertyQuery _propQ;

	protected ResourceIterator _rangeIter;
	protected ResourceIterator _propIter;

	// True if _rangeQ is a ClassVar and it is not locked by someone else
	protected boolean _rangeQisVar;

	// True if _propQ is a PropertyVar and it is not locked by someone else
	protected boolean _propQisVar;


	public RangeSelector(PropertyQuery propQ, ClassQuery rangeQ) {
		_rangeQ = rangeQ;
		_propQ = propQ;
	}


	public void initialize(RdfSchemaSource rss) throws QueryEvaluationException {
		_rangeQisVar = _rangeQ instanceof ClassVar &&
						! ((ClassVar)_rangeQ).hasValue();

		_propIter = _propQ.getProperties(rss);
		_propQisVar = (_propIter == null);

		if (_propIter == null) {
			_propIter = new StatementSubjectIterator(rss.getProperties());
		}
	}

	public boolean selectNext(RdfSchemaSource rss) throws QueryEvaluationException {
        org.openrdf.model.URI property;
        Value propRange;

		if (_rangeQisVar) {
			if (_rangeIter == null || !_rangeIter.hasNext()) {
				// Current range iterator exhausted
				while (_propIter.hasNext()) {
					property = (org.openrdf.model.URI)_propIter.next();

					Intersection intersect =
							new Intersection(rss.getRange(property, null));
					propRange = intersect.minimize(rss);

					if (propRange instanceof Intersection) {
						((ClassVar)_rangeQ).setValue(propRange);
					}
					else if (propRange instanceof Resource) {
						_rangeIter = new StatementSubjectIterator(rss.getSubClassOf(null, (Resource)propRange));
						// _rangeIter contains at least propRange because
						// subClassOf is reflexive
						if (_rangeIter.hasNext()) {
							((ClassVar)_rangeQ).setValue(_rangeIter.next());
						}
					}

					if (_propQisVar) {
						((PropertyVar)_propQ).setValue(property);
					}

					return true;
				}
			}

			// Subclasses of range class
			if (_rangeIter != null && _rangeIter.hasNext()) {
				Resource subClass = (Resource)_rangeIter.next();
				((ClassVar)_rangeQ).setValue(subClass);
				return true;
			}

			// No more results, release lock on vars
			((ClassVar)_rangeQ).setValue(null);
			if (_propQisVar) {
				((PropertyVar)_propQ).setValue(null);
			}

			return false;
		}
		else { // rangeQ is fixed
			// The range of the property itself, or one of its subclasses
			// must be equal to a value in _rangeQ

			boolean result = false;

			while (_propIter.hasNext()) {
				property = (org.openrdf.model.URI)_propIter.next();
				if (_propQisVar) {
					((PropertyVar)_propQ).setValue(property);
				}

				Intersection intersect =
						new Intersection(rss.getRange(property, null));
				propRange = intersect.minimize(rss);

				_rangeIter = _rangeQ.getClasses(rss);

				// _rangeIter (likely) contains just one value, as it
				// is either a ClassVar or a URI.
                Value nextRange;
				while (_rangeIter.hasNext()) {
					nextRange = _rangeIter.next();
					if (nextRange.equals(propRange)) {
						result = true;
						break;
					}
					else if (propRange instanceof Intersection) {
						if (nextRange instanceof Intersection) {
							result = ((Intersection)nextRange).lowerEqualThan((Intersection)propRange, rss);
							if (result == true) {
								break;
							}
						}
						else if (nextRange instanceof Resource) {
							Iterator intersectIter =
									((Intersection)propRange).getMembers().iterator();

							Resource member = null;
							boolean mismatch = false;
							while (intersectIter.hasNext()) {
								member = (Resource)intersectIter.next();
								if (!rss.isSubClassOf((Resource)nextRange, member)) {
									// not matched.
									mismatch = true;
									break;
								}
							}
							if (! mismatch) {
								result = true;
								break;
							}
						}
					}
					else if (nextRange instanceof Intersection) {
						Intersection myIntersection = new Intersection();
						myIntersection.add((Resource)propRange);
						result = ((Intersection)nextRange).lowerEqualThan(myIntersection, rss);
						if (result == true) {
							break;
						}
					}
					else if (nextRange instanceof Resource &&
						propRange instanceof Resource &&
						rss.isSubClassOf((Resource)nextRange, (Resource)propRange))
					{
						result = true;
						break;
					}
				}

				// Close the iterator to close the db connection
				_rangeIter.close();

				if (result == true) {
					if (!_propQisVar) {
						// _rangeQ and _propQ are both locked, return true only
						// once by closing the property iterator
						_propIter.close();
					}
					return true;
				}
			}

			// No more results, release lock on both vars
			if (_rangeQisVar) {
				((ClassVar)_rangeQ).setValue(null);
			}
			if (_propQisVar) {
				((PropertyVar)_propQ).setValue(null);
			}

			return false;
		}
	}

	public void clear() {
		if (_propIter != null) {
			// Close the iterators
			_propIter.close();

			if (_rangeIter != null) {
				_rangeIter.close();
			}

			// Release the lock on the variables
			if (_rangeQisVar) {
				((ClassVar)_rangeQ).setValue(null);
			}
			if (_propQisVar) {
				((PropertyVar)_propQ).setValue(null);
			}

			_propIter = null;
		}
	}

	public String toString() {
		StringBuffer result = new StringBuffer();
		result.append(_propQ.toString());
		result.append("{:");
		result.append(_rangeQ.toString());
		result.append("}");

		return result.toString();
	}
}
