/*  Sesame - Storage and Querying architecture for RDF and RDF Schema
 *  Copyright (C) 2001-2006 Aduna
 *
 *  Contact: 
 *  	Aduna
 *  	Prinses Julianaplein 14 b
 *  	3817 CS Amersfoort
 *  	The Netherlands
 *  	tel. +33 (0)33 465 99 87
 *  	fax. +33 (0)33 465 99 87
 *
 *  	http://aduna-software.com/
 *  	http://www.openrdf.org/
 *  
 *  This library is free software; you can redistribute it and/or
 *  modify it under the terms of the GNU Lesser General Public
 *  License as published by the Free Software Foundation; either
 *  version 2.1 of the License, or (at your option) any later version.
 *
 *  This library is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 *  Lesser General Public License for more details.
 *
 *  You should have received a copy of the GNU Lesser General Public
 *  License along with this library; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

package org.openrdf.sesame.query.rql.model;

import org.openrdf.model.Resource;

import org.openrdf.sesame.query.QueryEvaluationException;
import org.openrdf.sesame.sail.RdfSchemaSource;
import org.openrdf.sesame.sail.ResourceIterator;

public class PropertySelector implements Selector {

	protected PropertyVar _propVar;

	protected PropertyQuery _propQ;

	protected ResourceIterator _propQIter;

	// True if someone else locked this var.
	protected boolean _propVarLocked;

	/**
	 * Creates a new PropertySelector that fills the supplied propVar with
	 * values from propQ. propQ is not allowed to be an instance of
	 * PropertyVar.
	 * @exception IllegalArgumentException If propQ is an instance of
	 * PropertyVar.
	 **/
	public PropertySelector(PropertyVar propVar, PropertyQuery propQ) {
		if (propQ instanceof PropertyVar) {
			throw new IllegalArgumentException(
					"propQ must not be an instance of PropertyVar");
		}
		_propVar = propVar;
		_propQ = propQ;
	}

	public void initialize(RdfSchemaSource rss)
		throws QueryEvaluationException
	{
		_propQIter = _propQ.getProperties(rss);
		_propVarLocked = _propVar.hasValue();
	}

	public boolean selectNext(RdfSchemaSource rss)
		throws QueryEvaluationException
	{
		org.openrdf.model.URI propQValue;

		if (_propVarLocked) {
			// Check whether the value in _propVar is present in the
			// values of _propQ.
			Resource propVarValue = (Resource)_propVar.getValue();

			boolean result = false;

			while (_propQIter.hasNext()) {
				propQValue = (org.openrdf.model.URI)_propQIter.next();
				if (propVarValue.equals(propQValue)) {
					result = true;
				}
			}

			return result;
		}
		else { // property var not locked
			if (_propQIter.hasNext()) {
				propQValue = (org.openrdf.model.URI)_propQIter.next();
				_propVar.setValue(propQValue);
				return true;
			}

			// No more results, release the lock on _propVar
			_propVar.setValue(null);

			return false;
		}
	}

	public void clear() {
		if (_propQIter != null) {
			// Close the iterator
			_propQIter.close();

			// Release locks on vars
			if (!_propVarLocked) {
				_propVar.setValue(null);
			}
		}
	}

	public String toString() {
		StringBuffer result = new StringBuffer();
		result.append("(");
		result.append(_propQ.toString());
		result.append("{");
		result.append(_propVar.toString());
		result.append("})");

		return result.toString();
	}
}

