/*  Sesame - Storage and Querying architecture for RDF and RDF Schema
 *  Copyright (C) 2001-2006 Aduna
 *
 *  Contact: 
 *  	Aduna
 *  	Prinses Julianaplein 14 b
 *  	3817 CS Amersfoort
 *  	The Netherlands
 *  	tel. +33 (0)33 465 99 87
 *  	fax. +33 (0)33 465 99 87
 *
 *  	http://aduna-software.com/
 *  	http://www.openrdf.org/
 *  
 *  This library is free software; you can redistribute it and/or
 *  modify it under the terms of the GNU Lesser General Public
 *  License as published by the Free Software Foundation; either
 *  version 2.1 of the License, or (at your option) any later version.
 *
 *  This library is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 *  Lesser General Public License for more details.
 *
 *  You should have received a copy of the GNU Lesser General Public
 *  License along with this library; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

package org.openrdf.sesame.query.rql.model;

import org.openrdf.model.Literal;
import org.openrdf.model.Resource;
import org.openrdf.model.Statement;
import org.openrdf.model.Value;
import org.openrdf.model.impl.URIImpl;

import org.openrdf.sesame.query.QueryEvaluationException;
import org.openrdf.sesame.query.rql.model.iterators.StatementSubjectIterator;
import org.openrdf.sesame.sail.RdfSchemaSource;
import org.openrdf.sesame.sail.ResourceIterator;
import org.openrdf.sesame.sail.StatementIterator;
import org.openrdf.sesame.sail.ValueIterator;

public class InstanceSelector implements Selector {

	protected DataVar _dataVar;

	protected ClassQuery _classQ;

	protected ResourceIterator _classIter;

	protected ValueIterator _instIter;

	protected StatementIterator _statIter;

	protected boolean _firstIteration;

	// True if _dataVar was locked by someone else
	protected boolean _dataVarLocked;

	// True if _classQ is a ClassVar and it is not locked by someone else
	protected boolean _classQisVar;

	protected boolean _properInstances;

	public InstanceSelector(DataVar dataVar, ClassQuery classQ) {
		this(dataVar, classQ, false);
	}

	public InstanceSelector(DataVar dataVar, ClassQuery classQ,
			boolean properInstances)
	{
		_dataVar = dataVar;
		_classQ = classQ;
		_properInstances = properInstances;
	}

	public void initialize(RdfSchemaSource rss)
		throws QueryEvaluationException
	{
		_classIter = _classQ.getClasses(rss);

		_classQisVar = (_classIter == null);

		_dataVarLocked = _dataVar.hasValue();

		_firstIteration = true;
	}

	public boolean selectNext(RdfSchemaSource rss)
		throws QueryEvaluationException
	{
		Value dataValue = _dataVar.getValue();
		Resource classRes;

		if (_classQisVar) {
			// _classQ is an unlocked variable

			if (dataValue instanceof Literal) {
				if (_firstIteration) {
					((ClassVar)_classQ).setValue(URIImpl.RDFS_LITERAL);
					_firstIteration = false;
					return true;
				}
				else {
					((ClassVar)_classQ).setValue(null);
					return false;
				}
			}

			Resource dataRes = (Resource)dataValue;

			if (_statIter == null) {
				// First call, _statIter uninitialized
				if (_properInstances) {
					_statIter = rss.getDirectType(dataRes, null);
				}
				else {
					_statIter = rss.getType(dataRes, null);
				}
			}

			while (_statIter.hasNext()) {
				Statement stat = _statIter.next();

				if (stat.getObject() instanceof Resource) {
					((ClassVar)_classQ).setValue(stat.getObject());

					if (!_dataVarLocked) {
						_dataVar.setValue(stat.getSubject());
					}

					return true;
				}
				// else object is a Literal, try next statement
			}

			// No (more) results found

			_statIter.close();

			// Release locks
			((ClassVar)_classQ).setValue(null);
			if (!_dataVarLocked) {
				_dataVar.setValue(null);
			}

			return false;
		}
		else if (_dataVarLocked) {
			// Both _classQ and _dataVar are fixed/locked
			boolean isInstance = false;

			// Iterate over all classes until we have found one of which
			// dataValue is an instance
			while (_classIter.hasNext() && !isInstance) {
				classRes = _classIter.nextResource();
				isInstance = _isInstanceOf(dataValue, classRes, rss);
			}

			// Only return true or false once:
			_classIter.close();

			return isInstance;
		}
		else {
			// _dataVar is an unlocked variable, _classQ is not
			if (_instIter == null || !_instIter.hasNext()) {
				// Current instance iterator exhausted or not yet initialized
				if (_instIter != null) {
					_instIter.close();
				}

				while (_classIter.hasNext()) {
					classRes = _classIter.nextResource();

					if (classRes.equals(URIImpl.RDFS_LITERAL)) {
						_instIter = rss.getLiterals(null, null, null);
					}
					else if (classRes instanceof Intersection) {
						_instIter = ((Intersection)classRes).getInstances(rss);
					}
					else {
						if (_properInstances) {
							_instIter = new StatementSubjectIterator(
									rss.getDirectType(null, classRes));
						}
						else {
							_instIter = new StatementSubjectIterator(rss.getType(null,
									classRes));
						}
					}

					if (_instIter.hasNext()) {
						break;
					}
					else {
						// get instances of next class
						_instIter.close();
					}
				}
			}

			if (_instIter != null && _instIter.hasNext()) {
				_dataVar.setValue(_instIter.next());
				return true;
			}

			// Release lock
			_dataVar.setValue(null);
			return false;
		}
	}

	private boolean _isInstanceOf(Value dataValue, Resource classRes,
			RdfSchemaSource rss)
	{
		boolean result = false;

		if (dataValue instanceof Literal) {
			result = classRes.equals(URIImpl.RDFS_LITERAL);
		}
		else {
			Resource dataRes = (Resource)dataValue;

			if (classRes instanceof Intersection) {
				Intersection intersect = (Intersection)classRes;
				result = intersect.containsInstance(dataRes, rss);
			}
			else { // classRes is a 'normal' Resource
				if (_properInstances) {
					result = rss.isDirectType(dataRes, classRes);
				}
				else {
					result = rss.isType(dataRes, classRes);
				}
			}
		}

		return result;
	}

	public void clear() {
		if (_classIter != null) {
			// Close iterators
			_classIter.close();
			_classIter = null;
		}

		if (_statIter != null) {
			_statIter.close();
			_statIter = null;
		}

		// Release lock on vars
		if (!_dataVarLocked) {
			_dataVar.setValue(null);
			_dataVarLocked = true;
		}
		if (_classQisVar) {
			((ClassVar)_classQ).setValue(null);
			_classQisVar = false;
		}
	}

	public String toString() {
		StringBuffer result = new StringBuffer();
		result.append("{");
		result.append(_dataVar.toString());
		result.append(" : ");
		if (_properInstances) {
			result.append("^");
		}
		result.append(_classQ.toString());
		result.append("}");

		return result.toString();
	}
}
