/*  Sesame - Storage and Querying architecture for RDF and RDF Schema
 *  Copyright (C) 2001-2006 Aduna
 *
 *  Contact: 
 *  	Aduna
 *  	Prinses Julianaplein 14 b
 *  	3817 CS Amersfoort
 *  	The Netherlands
 *  	tel. +33 (0)33 465 99 87
 *  	fax. +33 (0)33 465 99 87
 *
 *  	http://aduna-software.com/
 *  	http://www.openrdf.org/
 *  
 *  This library is free software; you can redistribute it and/or
 *  modify it under the terms of the GNU Lesser General Public
 *  License as published by the Free Software Foundation; either
 *  version 2.1 of the License, or (at your option) any later version.
 *
 *  This library is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 *  Lesser General Public License for more details.
 *
 *  You should have received a copy of the GNU Lesser General Public
 *  License along with this library; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

package org.openrdf.sesame.query.rql.model;

import org.openrdf.model.Value;

import org.openrdf.sesame.query.QueryEvaluationException;
import org.openrdf.sesame.sail.RdfSchemaSource;
import org.openrdf.sesame.sail.ValueIterator;

public class In implements BooleanQuery {

/*-------------------------------------------+
| Variables								  |
+-------------------------------------------*/

	protected ResourceQuery _arg1;
	protected ResourceQuery _arg2;

/*-------------------------------------------+
| Constructors							   |
+-------------------------------------------*/

	public In(ResourceQuery arg1, ResourceQuery arg2) {
		_arg1 = arg1;
		_arg2 = arg2;
	}

/*-------------------------------------------+
| Methods									|
+-------------------------------------------*/

	public ResourceQuery getArg1() {
		return _arg1;
	}

	public ResourceQuery getArg2() {
		return _arg2;
	}

	/**
	 * Determines whether operand 1 is in the result set of operand 2.
	 *
	 * @param rss the repository abstraction layer which is to be queried
	 * during evaluation.
	 *
	 * @return true if operand 1 is in the result set of operand 2, false
	 * otherwise.
	 * @throws QueryEvaluationException
	 **/
	public boolean isTrue(RdfSchemaSource rss) throws QueryEvaluationException {
		// Get the one value of the first argument

		ValueIterator iter1 = _arg1.getResources(rss);

		if (iter1 == null) {
			throw new QueryEvaluationException("left operand is NULL in " +
					this.getQuery());
		}

		if (!iter1.hasNext()) {
			iter1.close();
			throw new QueryEvaluationException("left operand is missing in " +
					this.getQuery());
		}

		Value val1 = iter1.next();

		if (iter1.hasNext()) {
			iter1.close();
			throw new QueryEvaluationException(
					"left operand has multiple values " + this.getQuery());
		}

		iter1.close();

		// Iterate over the values of the second argument.

		ValueIterator iter2 = null;

		if (_arg2 instanceof URI) {
			// the second argument is a URI. This can only be interpreted
			// as meaning a class. 
			iter2 = ((URI)_arg2).getInstances(rss);
		}
		else {
			iter2 = _arg2.getResources(rss);
		}

		if (iter2 == null) {
			throw new QueryEvaluationException("right operand is NULL in " +
					this.getQuery());
		}

		boolean result = false;

		Value val2;

		// FIXME instead of this check for SfwQuery, it would be better to
		// let all ResourceQueries return Projection objects.
		if (_arg2 instanceof SfwQuery) {
			Projection projection;

			while (iter2.hasNext()) {
				projection = (Projection)iter2.next();

				if (projection.size() > 1) {
					iter2.close();
					throw new QueryEvaluationException("right operand of IN may only contain one variable.");
				}

				// projection has only one variable.
				val2 = projection.get(0);

				if (val1.equals(val2)) {
					result = true;
					break;
				}
			}
		}
		else {
			// not an sfwquery, so normal resources in the iterator.
			while (iter2.hasNext()) {
				val2 = iter2.next();

				if (val1.equals(val2)) {
					result = true;
					break;
				}
			}
		}

		// Close the iterator
		iter2.close();

		return result;
	}

	public String getQuery() {
		return this.toString();
	}

	public String toString() {
		return _arg1.toString() + " in " + _arg2.toString();
	}
}
