/*  Sesame - Storage and Querying architecture for RDF and RDF Schema
 *  Copyright (C) 2001-2006 Aduna
 *
 *  Contact: 
 *  	Aduna
 *  	Prinses Julianaplein 14 b
 *  	3817 CS Amersfoort
 *  	The Netherlands
 *  	tel. +33 (0)33 465 99 87
 *  	fax. +33 (0)33 465 99 87
 *
 *  	http://aduna-software.com/
 *  	http://www.openrdf.org/
 *  
 *  This library is free software; you can redistribute it and/or
 *  modify it under the terms of the GNU Lesser General Public
 *  License as published by the Free Software Foundation; either
 *  version 2.1 of the License, or (at your option) any later version.
 *
 *  This library is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 *  Lesser General Public License for more details.
 *
 *  You should have received a copy of the GNU Lesser General Public
 *  License along with this library; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */
package org.openrdf.sesame.query.rql.model;

import java.util.Iterator;

import org.openrdf.model.Resource;
import org.openrdf.model.Value;

import org.openrdf.sesame.query.QueryEvaluationException;
import org.openrdf.sesame.query.rql.model.iterators.StatementSubjectIterator;
import org.openrdf.sesame.sail.RdfSchemaSource;
import org.openrdf.sesame.sail.ResourceIterator;

public class DomainSelector implements Selector {

	protected ClassQuery _domainQ;

	protected PropertyQuery _propQ;

	protected ResourceIterator _domainIter;

	protected ResourceIterator _propIter;

	// True if _domainQ is a ClassVar and it is not locked by someone else
	protected boolean _domainQisVar;

	// True if _propQ is a PropertyVar and it is not locked by someone else
	protected boolean _propQisVar;

	public DomainSelector(ClassQuery domainQ, PropertyQuery propQ) {
		_domainQ = domainQ;
		_propQ = propQ;
	}

	public void initialize(RdfSchemaSource rss)
		throws QueryEvaluationException
	{
		_domainQisVar = _domainQ instanceof ClassVar
				&& !((ClassVar)_domainQ).hasValue();

		_propIter = _propQ.getProperties(rss);
		_propQisVar = (_propIter == null);

		if (_propIter == null) {
			_propIter = new StatementSubjectIterator(rss.getProperties());
		}
	}

	public boolean selectNext(RdfSchemaSource rss)
		throws QueryEvaluationException
	{
		org.openrdf.model.URI property;
		Value propDomain;

		if (_domainQisVar) {
			if (_domainIter == null || !_domainIter.hasNext()) {
				// Current domain iterator exhausted
				while (_propIter.hasNext()) {
					property = (org.openrdf.model.URI)_propIter.next();

					Intersection intersect = new Intersection(rss.getDomain(
							property, null));
					propDomain = intersect.minimize(rss);

					if (propDomain instanceof Intersection) {
						((ClassVar)_domainQ).setValue(propDomain);
					}
					else if (propDomain instanceof Resource) {
						_domainIter = new StatementSubjectIterator(rss.getSubClassOf(
								null, (Resource)propDomain));
						// _domainIter contains at least propDomain because
						// subClassOf is reflexive
						if (_domainIter.hasNext()) {
							((ClassVar)_domainQ).setValue(_domainIter.next());
						}
					}

					if (_propQisVar) {
						((PropertyVar)_propQ).setValue(property);
					}

					return true;
				}
			}

			// Subclasses of domain class
			if (_domainIter != null && _domainIter.hasNext()) {
				Resource subClass = (Resource)_domainIter.next();
				((ClassVar)_domainQ).setValue(subClass);
				return true;
			}

			// No more results, release lock on vars
			((ClassVar)_domainQ).setValue(null);
			if (_propQisVar) {
				((PropertyVar)_propQ).setValue(null);
			}

			return false;
		}
		else { // domainQ is fixed
			// The domain of the property itself, or one of its subclasses
			// must be equal to a value in _domainQ

			boolean result = false;

			while (_propIter.hasNext()) {
				property = (org.openrdf.model.URI)_propIter.next();
				if (_propQisVar) {
					((PropertyVar)_propQ).setValue(property);
				}

				Intersection intersect = new Intersection(rss.getDomain(property,
						null));
				propDomain = intersect.minimize(rss);

				_domainIter = _domainQ.getClasses(rss);

				// _domainIter (likely) contains just one value, as it
				// is either a ClassVar or a URI.
				Value nextDomain;
				while (_domainIter.hasNext()) {
					nextDomain = _domainIter.next();
					if (nextDomain.equals(propDomain)) {
						result = true;
						break;
					}
					else if (propDomain instanceof Intersection) {
						if (nextDomain instanceof Intersection) {
							result = ((Intersection)nextDomain).lowerEqualThan(
									(Intersection)propDomain, rss);
							if (result == true) {
								break;
							}
						}
						else if (nextDomain instanceof Resource) {
							Iterator intersectIter = ((Intersection)propDomain).getMembers().iterator();

							Resource member = null;
							boolean mismatch = false;
							while (intersectIter.hasNext()) {
								member = (Resource)intersectIter.next();
								if (!rss.isSubClassOf((Resource)nextDomain, member)) {
									// not matched.
									mismatch = true;
									break;
								}
							}
							if (!mismatch) {
								result = true;
								break;
							}
						}
					}
					else if (nextDomain instanceof Intersection) {
						Intersection myIntersection = new Intersection();
						myIntersection.add((Resource)propDomain);
						result = ((Intersection)nextDomain).lowerEqualThan(
								myIntersection, rss);
						if (result == true) {
							break;
						}
					}
					else if (nextDomain instanceof Resource
							&& propDomain instanceof Resource
							&& rss.isSubClassOf((Resource)nextDomain,
									(Resource)propDomain))
					{
						result = true;
						break;
					}
				}

				// Close the iterator to close the db connection
				_domainIter.close();

				if (result == true) {
					if (!_propQisVar) {
						// _domainQ and _propQ are both locked, return true only
						// once by closing the property iterator
						_propIter.close();
					}
					return true;
				}
			}

			// No more results, release lock on both vars
			if (_domainQisVar) {
				((ClassVar)_domainQ).setValue(null);
			}
			if (_propQisVar) {
				((PropertyVar)_propQ).setValue(null);
			}

			return false;
		}
	}

	public void clear() {
		if (_propIter != null) {
			// Close the iterators
			_propIter.close();

			if (_domainIter != null) {
				_domainIter.close();
			}

			// Release the lock on the variables
			if (_domainQisVar) {
				((ClassVar)_domainQ).setValue(null);
			}
			if (_propQisVar) {
				((PropertyVar)_propQ).setValue(null);
			}

			_propIter = null;
		}
	}

	public String toString() {
		StringBuffer result = new StringBuffer();
		result.append("{:");
		result.append(_domainQ.toString());
		result.append("}");
		result.append(_propQ.toString());

		return result.toString();
	}
}
