/*  Sesame - Storage and Querying architecture for RDF and RDF Schema
 *  Copyright (C) 2001-2006 Aduna
 *
 *  Contact: 
 *  	Aduna
 *  	Prinses Julianaplein 14 b
 *  	3817 CS Amersfoort
 *  	The Netherlands
 *  	tel. +33 (0)33 465 99 87
 *  	fax. +33 (0)33 465 99 87
 *
 *  	http://aduna-software.com/
 *  	http://www.openrdf.org/
 *  
 *  This library is free software; you can redistribute it and/or
 *  modify it under the terms of the GNU Lesser General Public
 *  License as published by the Free Software Foundation; either
 *  version 2.1 of the License, or (at your option) any later version.
 *
 *  This library is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 *  Lesser General Public License for more details.
 *
 *  You should have received a copy of the GNU Lesser General Public
 *  License along with this library; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

package org.openrdf.sesame.query.rql.model;

import java.io.IOException;

import org.openrdf.model.Resource;
import org.openrdf.model.Value;

import org.openrdf.sesame.query.QueryEvaluationException;
import org.openrdf.sesame.query.TableQueryResultListener;
import org.openrdf.sesame.sail.RdfSchemaSource;
import org.openrdf.sesame.sail.ResourceIterator;
import org.openrdf.sesame.sail.ValueIterator;

public class Domain implements ClassQuery {

	protected PropertyQuery _propQ;

	public Domain(PropertyQuery propQ) {
		_propQ = propQ;
	}

	public void evaluate(RdfSchemaSource rss, TableQueryResultListener listener)
		throws QueryEvaluationException
	{
		ResourceIterator iter = getClasses(rss);

		try {
			String columnHeaders[] = new String[1];
			columnHeaders[0] = this.getQuery();
			listener.startTableQueryResult(columnHeaders);

			iter = getClasses(rss);

			while (iter.hasNext()) {
				listener.startTuple();
				listener.tupleValue(iter.next());
				listener.endTuple();
			}
			iter.close();
			listener.endTableQueryResult();
		}
		catch (IOException e) {
			iter.close();
			throw new QueryEvaluationException(e);
		}
	}

	public ValueIterator getResources(RdfSchemaSource rss)
		throws QueryEvaluationException
	{
		return getClasses(rss);
	}

	public ResourceIterator getClasses(RdfSchemaSource rss)
		throws QueryEvaluationException
	{
		ResourceIterator propIter = _propQ.getProperties(rss);
		return new Iterator(rss, propIter);
	}

	/*
	 * Notice that, although a domain may contain more than one class
	 * name, these classes are always returned as just one object (either
	 * a Resource or an Intersection).
	 */
	public boolean returnsSet() {
		return false;
	}

	public String getQuery() {
		return "domain(" + _propQ.toString() + ")";
	}

	public String toString() {
		return "domain(" + _propQ.toString() + ")";
	}

	/*---------------------------------------------------+
	 | Iterator                                           |
	 +---------------------------------------------------*/

	static class Iterator implements ResourceIterator {

		protected RdfSchemaSource _rss;

		protected ResourceIterator _propIter;

		public Iterator(RdfSchemaSource rss, ResourceIterator propIter) {
			super();
			_rss = rss;
			_propIter = propIter;
		}

		public boolean hasNext() throws QueryEvaluationException {
			return _propIter.hasNext();
		}

		public Value next()
			throws QueryEvaluationException
		{
			Resource property = (Resource)_propIter.next();

			Intersection intersect = new Intersection(_rss.getDomain(property,
					null));

			return intersect.minimize(_rss);
		}

		public Resource nextResource()
			throws QueryEvaluationException
		{
			return (Resource)next();
		}

		public void close() {
			_propIter.close();
		}

		protected void finalize() {
			close();
		}
	}
}
