/*  Sesame - Storage and Querying architecture for RDF and RDF Schema
 *  Copyright (C) 2001-2006 Aduna
 *
 *  Contact: 
 *  	Aduna
 *  	Prinses Julianaplein 14 b
 *  	3817 CS Amersfoort
 *  	The Netherlands
 *  	tel. +33 (0)33 465 99 87
 *  	fax. +33 (0)33 465 99 87
 *
 *  	http://aduna-software.com/
 *  	http://www.openrdf.org/
 *  
 *  This library is free software; you can redistribute it and/or
 *  modify it under the terms of the GNU Lesser General Public
 *  License as published by the Free Software Foundation; either
 *  version 2.1 of the License, or (at your option) any later version.
 *
 *  This library is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 *  Lesser General Public License for more details.
 *
 *  You should have received a copy of the GNU Lesser General Public
 *  License along with this library; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

package org.openrdf.sesame.query.rql.model;

import org.openrdf.model.Resource;
import org.openrdf.model.Statement;
import org.openrdf.model.Value;
import org.openrdf.model.impl.URIImpl;

import org.openrdf.sesame.sail.RdfSchemaSource;
import org.openrdf.sesame.sail.ResourceIterator;
import org.openrdf.sesame.sail.StatementIterator;
import org.openrdf.sesame.sail.util.EmptyStatementIterator;

public class DataPathSelector implements Selector {

	protected DataVar _sourceVar;
	protected PropertyVar _propVar;
	protected DataVar _targetVar;

	protected boolean _propIsExtended;

	protected ResourceIterator _propertyIter;
	protected StatementIterator _statementIter;

	protected boolean _sourceVarLocked;
	protected boolean _propVarLocked;
	protected boolean _targetVarLocked;

	protected boolean _sourceAndTargetAreEqual;


	/**
	 * Creates a new DataPathSelector. The parameter 'propIsExtended'
	 * determines whether an extended interpretation of the property value
	 * in 'propVar' should be taken. Extended interpretation means that
	 * the property's sub properties will also be searched for.
	 **/
	public DataPathSelector(DataVar sourceVar, PropertyVar propVar,
			DataVar targetVar, boolean propIsExtended)
	{
		_sourceVar = sourceVar;
		_propVar = propVar;
		_targetVar = targetVar;
		_propIsExtended = propIsExtended;
	}

	public PropertyVar getPropertyVar() {
		return _propVar;
	}

	public DataVar getSourceVar() {
		return _sourceVar;
	}

	public DataVar getTargetVar() {
		return _targetVar;
	}

	public void initialize(RdfSchemaSource rss) {
		_sourceVarLocked = _sourceVar.hasValue();
		_propVarLocked = _propVar.hasValue();
		_targetVarLocked = _targetVar.hasValue();

		// Special case: sourceVar and targetVar could be one and the
		// same variable, or they could have been implicitely joined.
		if (!_sourceVarLocked && !_targetVarLocked) {
			// Use RDF_TYPE as dummy value:
			_sourceVar.setValue(URIImpl.RDF_TYPE);
			_sourceAndTargetAreEqual = _targetVar.hasValue();
			_sourceVar.setValue(null);
		}

		try {
			Resource sourceValue = (Resource)_sourceVar.getValue();
			org.openrdf.model.URI propValue = (org.openrdf.model.URI)_propVar.getValue();
			Value targetValue = _targetVar.getValue();

			if (_propIsExtended) {
				_statementIter = rss.getStatements(sourceValue, propValue, targetValue);
			} else {
				_statementIter = rss.getExplicitStatements(sourceValue, propValue, targetValue);
			}
		}
		catch (ClassCastException e) {
			// source value is not a Resource, or property value is not a URI,
			// possibly due to a wrong comparison in the where clause.
			_statementIter = new EmptyStatementIterator();
		}
	}

	public boolean selectNext(RdfSchemaSource rss) {
		while (_statementIter.hasNext()) {
			Statement st = _statementIter.next();

			if (_sourceAndTargetAreEqual) {
				// subject and object should be equal
				if (!st.getSubject().equals(st.getObject())) {
					continue;
				}
			}

			if (!_sourceVarLocked) {
				_sourceVar.setValue(st.getSubject());
			}
			if (!_propVarLocked) {
				_propVar.setValue(st.getPredicate());
			}
			if (!_targetVarLocked) {
				_targetVar.setValue(st.getObject());
			}

			return true;
		}

		// No more results, release locks
		if (!_sourceVarLocked) {
			_sourceVar.setValue(null);
		}
		if (!_propVarLocked) {
			_propVar.setValue(null);
		}
		if (!_targetVarLocked) {
			_targetVar.setValue(null);
		}

		return false;
	}

	public void clear() {
		if (_statementIter != null) {
			// Close the statement iterator
			_statementIter.close();

			// Release locks on vars
			if (!_sourceVarLocked) {
				_sourceVar.setValue(null);
			}
			if (!_propVarLocked) {
				_propVar.setValue(null);
			}
			if (!_targetVarLocked) {
				_targetVar.setValue(null);
			}
		}
	}

	public String toString() {
		StringBuffer result = new StringBuffer();
		result.append("{");
		if (_sourceVar != null) {
			result.append(_sourceVar.toString());
		}
		result.append("}");
		if (!_propIsExtended) {
			result.append("^");
		}
		result.append(_propVar.toString());
		result.append("{");
		if (_targetVar != null) {
			result.append(_targetVar.toString());
		}
		result.append("}");
		return result.toString();
	}
}
