/*  Sesame - Storage and Querying architecture for RDF and RDF Schema
 *  Copyright (C) 2001-2006 Aduna
 *
 *  Contact: 
 *  	Aduna
 *  	Prinses Julianaplein 14 b
 *  	3817 CS Amersfoort
 *  	The Netherlands
 *  	tel. +33 (0)33 465 99 87
 *  	fax. +33 (0)33 465 99 87
 *
 *  	http://aduna-software.com/
 *  	http://www.openrdf.org/
 *  
 *  This library is free software; you can redistribute it and/or
 *  modify it under the terms of the GNU Lesser General Public
 *  License as published by the Free Software Foundation; either
 *  version 2.1 of the License, or (at your option) any later version.
 *
 *  This library is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 *  Lesser General Public License for more details.
 *
 *  You should have received a copy of the GNU Lesser General Public
 *  License along with this library; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

package org.openrdf.sesame.query.rql.model;

import org.openrdf.model.Resource;
import org.openrdf.model.Value;

import org.openrdf.sesame.query.QueryEvaluationException;
import org.openrdf.sesame.sail.RdfSchemaSource;
import org.openrdf.sesame.sail.ResourceIterator;

public class ClassSelector implements Selector {

	protected ClassVar _classVar;

	protected ClassQuery _classQ;

	protected ResourceIterator _classQIter;

	// True if someone else locked this var.
	protected boolean _classVarLocked;

	/**
	 * Creates a new ClassSelector that fills the supplied classVar with
	 * values from classQ. classQ is not allowed to be an instance of
	 * ClassVar.
	 * @exception IllegalArgumentException If classQ is an instance of
	 * ClassVar.
	 **/
	public ClassSelector(ClassVar classVar, ClassQuery classQ) {
		if (classQ instanceof ClassVar) {
			throw new IllegalArgumentException(
					"classQ must not be an instance of ClassVar");
		}
		_classVar = classVar;
		_classQ = classQ;
	}

	public void initialize(RdfSchemaSource rss)
		throws QueryEvaluationException
	{
		_classQIter = _classQ.getClasses(rss);
		_classVarLocked = (_classVar.getValue() != null);
	}

	public boolean selectNext(RdfSchemaSource rss)
		throws QueryEvaluationException
	{
		Resource classQValue;

		if (_classVarLocked) {
			// Check whether the value in _classVar is present in the
			// values of _classQ.
			Value classVarValue = _classVar.getValue();

			boolean result = false;

			while (_classQIter.hasNext()) {
				classQValue = (Resource)_classQIter.next();

				if (classVarValue instanceof Resource) {
					if (classVarValue.equals(classQValue)) {
						result = true;
					}
				}
				else { // Intersection (caused by a join on a property probably)
					// FIXME this needs some "complicated" inferencing.
					result = false;
				}
			}

			return result;
		}
		else { // class var not locked
			if (_classQIter.hasNext()) {
				classQValue = (Resource)_classQIter.next();
				_classVar.setValue(classQValue);
				return true;
			}

			// No more results, release the lock on _classVar
			_classVar.setValue(null);

			return false;
		}
	}

	public void clear() {
		if (_classQIter != null) {
			// Close the iterator
			_classQIter.close();

			// Release the lock on the variable
			if (!_classVarLocked) {
				_classVar.setValue(null);
			}
		}
	}

	public String toString() {
		StringBuffer result = new StringBuffer();
		result.append("(");
		result.append(_classQ.toString());
		result.append("{");
		result.append(_classVar.toString());
		result.append("})");

		return result.toString();
	}
}

