/*  Sesame - Storage and Querying architecture for RDF and RDF Schema
 *  Copyright (C) 2001-2006 Aduna
 *
 *  Contact: 
 *  	Aduna
 *  	Prinses Julianaplein 14 b
 *  	3817 CS Amersfoort
 *  	The Netherlands
 *  	tel. +33 (0)33 465 99 87
 *  	fax. +33 (0)33 465 99 87
 *
 *  	http://aduna-software.com/
 *  	http://www.openrdf.org/
 *  
 *  This library is free software; you can redistribute it and/or
 *  modify it under the terms of the GNU Lesser General Public
 *  License as published by the Free Software Foundation; either
 *  version 2.1 of the License, or (at your option) any later version.
 *
 *  This library is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 *  Lesser General Public License for more details.
 *
 *  You should have received a copy of the GNU Lesser General Public
 *  License along with this library; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

package org.openrdf.sesame.query.rql.model;

import java.io.IOException;
import java.util.NoSuchElementException;

import org.openrdf.model.Resource;
import org.openrdf.model.Value;

import org.openrdf.sesame.query.QueryEvaluationException;
import org.openrdf.sesame.query.TableQueryResultListener;
import org.openrdf.sesame.sail.RdfSchemaSource;
import org.openrdf.sesame.sail.ResourceIterator;
import org.openrdf.sesame.sail.ValueIterator;

public class ArraySelection implements ClassQuery, PropertyQuery, DataQuery {

	protected ResourceQuery _resourceQ;

	protected int _startIdx;

	protected int _endIdx;

	public ArraySelection(ResourceQuery resourceQ, int startIdx, int endIdx) {
		_resourceQ = resourceQ;
		_startIdx = startIdx;
		_endIdx = endIdx;
	}

	public void evaluate(RdfSchemaSource rss, TableQueryResultListener listener)
			throws QueryEvaluationException {
		ValueIterator iter = null;

		try {
			listener.startTableQueryResult();

			iter = getResources(rss);
			Value value;
			while (iter.hasNext()) {
				value = (Value)iter.next();
				listener.startTuple();
				listener.tupleValue(value);
				listener.endTuple();
			}

			listener.endTableQueryResult();
		}
		catch (IOException e) {
			throw new QueryEvaluationException(e);
		}
		finally {
			if (iter != null) {
				iter.close();
			}
		}
	}

	public ValueIterator getResources(RdfSchemaSource rss) throws QueryEvaluationException {
		return new ValIterator(_resourceQ.getResources(rss));
	}

	public ResourceIterator getClasses(RdfSchemaSource rss)
			throws QueryEvaluationException {
		if (_resourceQ instanceof ClassQuery) {
			return new ResIterator(((ClassQuery)_resourceQ).getClasses(rss));
		}

		throw new QueryEvaluationException("Not a ClassQuery: "
				+ _resourceQ.getQuery());
	}

	public ResourceIterator getProperties(RdfSchemaSource rss)
			throws QueryEvaluationException {
		if (_resourceQ instanceof PropertyQuery) {
			return new ResIterator(((PropertyQuery)_resourceQ).getProperties(rss));
		}

		throw new QueryEvaluationException("Not a PropertyQuery: "
				+ _resourceQ.getQuery());
	}

	public ValueIterator getInstances(RdfSchemaSource rss)
			throws QueryEvaluationException {
		if (_resourceQ instanceof DataQuery) {
			return getResources(rss);
		}

		throw new QueryEvaluationException("Not a DataQuery: "
				+ _resourceQ.getQuery());
	}

	public boolean returnsSet() {
		return _endIdx > _startIdx;
	}

	public String getQuery() {
		return this.toString();
	}

	public String toString() {
		StringBuffer result = new StringBuffer();
		result.append(_resourceQ.toString());
		result.append(" [");
		result.append(_startIdx);
		result.append(", ");
		result.append(_endIdx);
		result.append("]");
		return result.toString();
	}

	/*---------------------------------------------------+
	 | Iterator                                           |
	 +---------------------------------------------------*/

	class ValIterator implements ValueIterator {

		protected int _curIndex;

		protected ValueIterator _sourceArray;

		public ValIterator(ValueIterator sourceArray) throws QueryEvaluationException {
			_sourceArray = sourceArray;
			_curIndex = 1;

			// Discard the first _startIdx elements.
			while (sourceArray.hasNext() && _curIndex < _startIdx) {
				sourceArray.next();
				_curIndex++;
			}
		}

		public boolean hasNext() throws QueryEvaluationException {
			return (_curIndex <= _endIdx && _sourceArray.hasNext());
		}

		public Value next() throws QueryEvaluationException {
			if (hasNext()) {
				_curIndex++;
				return _sourceArray.next();
			}
			throw new NoSuchElementException();
		}

		public void close() {
			_curIndex = _endIdx + 1;
			_sourceArray.close();
		}

		protected void finalize() {
			close();
		}
	}

	class ResIterator extends ValIterator implements ResourceIterator {

		public ResIterator(ResourceIterator iter) throws QueryEvaluationException {
			super(iter);
		}

		public Resource nextResource() throws QueryEvaluationException {
			return (Resource)next();
		}
	}
}
