/*  Sesame - Storage and Querying architecture for RDF and RDF Schema
 *  Copyright (C) 2001-2006 Aduna
 *
 *  Contact: 
 *  	Aduna
 *  	Prinses Julianaplein 14 b
 *  	3817 CS Amersfoort
 *  	The Netherlands
 *  	tel. +33 (0)33 465 99 87
 *  	fax. +33 (0)33 465 99 87
 *
 *  	http://aduna-software.com/
 *  	http://www.openrdf.org/
 *  
 *  This library is free software; you can redistribute it and/or
 *  modify it under the terms of the GNU Lesser General Public
 *  License as published by the Free Software Foundation; either
 *  version 2.1 of the License, or (at your option) any later version.
 *
 *  This library is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 *  Lesser General Public License for more details.
 *
 *  You should have received a copy of the GNU Lesser General Public
 *  License along with this library; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

package org.openrdf.sesame.query.rql;

import java.io.IOException;
import java.io.StringReader;

import org.openrdf.util.StringUtil;

import org.openrdf.model.impl.LiteralImpl;

import org.openrdf.sesame.query.MalformedQueryException;
import org.openrdf.sesame.query.QueryEvaluationException;
import org.openrdf.sesame.query.TableQueryResultListener;
import org.openrdf.sesame.query.rql.model.BooleanQuery;
import org.openrdf.sesame.query.rql.model.Query;
import org.openrdf.sesame.query.rql.model.ResourceQuery;
import org.openrdf.sesame.query.rql.parser.ParseException;
import org.openrdf.sesame.query.rql.parser.RqlParser;
import org.openrdf.sesame.query.rql.parser.TokenMgrError;
import org.openrdf.sesame.sail.RdfSchemaSource;

/**
 * QueryEngine for RQL queries.
 **/
public class RqlEngine {

/*---------------+
| Variables      |
+---------------*/

	/**
	 * RDF Schema aware (read-only) repository
	 **/
	protected RdfSchemaSource _rss;

/*---------------+
| Constructors   |
+---------------*/

	/**
	 * Creates a new QueryEngine that will use the supplied RdfSchemaSource
	 * for retrieving the data.
	 *
	 * @exception IllegalArgumentException If the supplied RdfSchemaSource is
	 * equal to null.
	 **/
	public RqlEngine(RdfSchemaSource rss) {
		if (rss == null) {
			throw new IllegalArgumentException("rss must not be nul");
		}
		_rss = rss;
	}

/*---------------+
| Methods        |
+---------------*/

	/**
	 * Evaluates the supplied Query and reports the results to the
	 * supplied TableQueryResultListener.
	 *
	 * @exception IOException If an I/O error occurred during the writing
	 * of the results.
	 * @throws QueryEvaluationException If an error occurred during query
	 * evaluation.
	 **/
	public void evaluateQuery(Query query, TableQueryResultListener resList)
		throws IOException, QueryEvaluationException
	{   
		query = RqlOptimizer.optimize(query);

		if (query instanceof ResourceQuery) {
			ResourceQuery resourceQuery = (ResourceQuery)query;
			resourceQuery.evaluate(_rss, resList);
		}
		else if (query instanceof BooleanQuery) {
			BooleanQuery booleanQuery = (BooleanQuery)query;
			boolean result = booleanQuery.isTrue(_rss);

			//FIXME: this is not the way to do this
			resList.startTableQueryResult();
			resList.startTuple();
			resList.tupleValue(new LiteralImpl(String.valueOf(result)));
			resList.endTuple();
			resList.endTableQueryResult();
		}
	}

	/**
	 * Parses the supplied RQL query and builds a query model from it.
	 * @return The query model.
	 *
	 * @exception MalformedQueryException If the query was syntactically
	 * incorrect.
	 **/
	public Query parseQuery(String rqlQuery)
		throws MalformedQueryException
	{
		rqlQuery = _swapUsingNamespace(rqlQuery);

		RqlParser rqlParser = new RqlParser( new StringReader(rqlQuery) );

		try {
			return rqlParser.parse();
		}
		catch (TokenMgrError tme) {
			throw new MalformedQueryException(tme.getMessage());
		}
		catch (ParseException pe) {
			throw new MalformedQueryException(pe);
		}
	}

	private String _swapUsingNamespace(String query) {
		String queryLC = query.toLowerCase();

		int usingIdx = StringUtil.lastIndexOfWord(queryLC, "using");

		String result;

		if (usingIdx != -1) {
			// Swap the part before and after the 'using' keyword
			String part1 = query.substring(0, usingIdx);
			String part2 = query.substring(usingIdx);

			result = part2 + "\n" + part1;
		}
		else {
			// No 'using' clause
			result = query;
		}

		return result;
	}
}
