/*  Sesame - Storage and Querying architecture for RDF and RDF Schema
 *  Copyright (C) 2001-2006 Aduna
 *
 *  Contact: 
 *  	Aduna
 *  	Prinses Julianaplein 14 b
 *  	3817 CS Amersfoort
 *  	The Netherlands
 *  	tel. +33 (0)33 465 99 87
 *  	fax. +33 (0)33 465 99 87
 *
 *  	http://aduna-software.com/
 *  	http://www.openrdf.org/
 *  
 *  This library is free software; you can redistribute it and/or
 *  modify it under the terms of the GNU Lesser General Public
 *  License as published by the Free Software Foundation; either
 *  version 2.1 of the License, or (at your option) any later version.
 *
 *  This library is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 *  Lesser General Public License for more details.
 *
 *  You should have received a copy of the GNU Lesser General Public
 *  License along with this library; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

package org.openrdf.sesame.query.rdql;

import java.io.IOException;
import java.io.StringReader;

import org.openrdf.util.StringUtil;

import org.openrdf.sesame.query.MalformedQueryException;
import org.openrdf.sesame.query.QueryEvaluationException;
import org.openrdf.sesame.query.TableQuery;
import org.openrdf.sesame.query.TableQueryResultListener;
import org.openrdf.sesame.query.rdql.parser.ParseException;
import org.openrdf.sesame.query.rdql.parser.RdqlParser;
import org.openrdf.sesame.query.rdql.parser.TokenMgrError;
import org.openrdf.sesame.sail.RdfSource;

/**
 * QueryEngine for RDQL queries.
 **/
public class RdqlEngine {

/*---------------+
| Variables      |
+---------------*/

	/**
	 * RDF (read-only) repository
	 **/
	protected RdfSource _rdfSource;

/*---------------+
| Constructors   |
+---------------*/

	/**
	 * Creates a new QueryEngine that will use the supplied RdfSource
	 * for retrieving the data.
	 *
	 * @exception IllegalArgumentException If the supplied RdfSource is
	 * equal to null.
	 **/
	public RdqlEngine(RdfSource rdfSource) {
		if (rdfSource == null) {
			throw new IllegalArgumentException("rdfSource must not be nul");
		}
		_rdfSource = rdfSource;
	}

/*---------------+
| Methods        |
+---------------*/

	/**
	 * Evaluates the supplied TableQuery and reports the results to the
	 * supplied TableQueryResultListener.
	 *
	 * @throws IOException If an I/O error occurred during the writing
	 * of the results.
	 * @throws QueryEvaluationException If an error occurred during query
	 * evaluation.
	 **/
	public void evaluateQuery(TableQuery tableQuery, TableQueryResultListener tableListener)
		throws IOException, QueryEvaluationException
	{   
		tableQuery.optimize(_rdfSource);
		tableQuery.evaluate(_rdfSource, tableListener);
	}

	/**
	 * Parses the supplied RDQL query and builds a query model from it.
	 * @return The query model.
	 *
	 * @exception MalformedQueryException If the query was syntactically
	 * incorrect.
	 **/
	public TableQuery parseQuery(String rdqlQuery)
		throws MalformedQueryException
	{
		rdqlQuery = _swapUsing(rdqlQuery);

		RdqlParser rdqlParser = new RdqlParser( new StringReader(rdqlQuery) );

		try {
			return rdqlParser.parseQuery();
		}
		catch (TokenMgrError tme) {
			throw new MalformedQueryException(tme.getMessage());
		}
		catch (ParseException pe) {
			throw new MalformedQueryException(pe);
		}
	}

	private String _swapUsing(String query) {
		String queryLC = query.toLowerCase();

		// Check for presence of 'using' clause
		int usingIdx = StringUtil.lastIndexOfWord(queryLC, "using");

		String result;

		if (usingIdx != -1) {
			// Swap the part before and after the 'using' keyword
			String part1 = query.substring(0, usingIdx);
			String part2 = query.substring(usingIdx);

			result = part2 + "\n" + part1;
		}
		else {
			// No 'using' clause
			result = query;
		}

		return result;
	}
}
