/*  Sesame - Storage and Querying architecture for RDF and RDF Schema
 *  Copyright (C) 2001-2006 Aduna
 *
 *  Contact: 
 *  	Aduna
 *  	Prinses Julianaplein 14 b
 *  	3817 CS Amersfoort
 *  	The Netherlands
 *  	tel. +33 (0)33 465 99 87
 *  	fax. +33 (0)33 465 99 87
 *
 *  	http://aduna-software.com/
 *  	http://www.openrdf.org/
 *  
 *  This library is free software; you can redistribute it and/or
 *  modify it under the terms of the GNU Lesser General Public
 *  License as published by the Free Software Foundation; either
 *  version 2.1 of the License, or (at your option) any later version.
 *
 *  This library is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 *  Lesser General Public License for more details.
 *
 *  You should have received a copy of the GNU Lesser General Public
 *  License along with this library; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */
package org.openrdf.sesame.query;

import java.io.IOException;
import java.io.OutputStream;
import java.io.Writer;
import java.util.Iterator;

import org.openrdf.util.xml.XmlWriter;

import org.openrdf.model.BNode;
import org.openrdf.model.Literal;
import org.openrdf.model.Resource;
import org.openrdf.model.URI;
import org.openrdf.model.Value;
import org.openrdf.model.impl.URIImpl;

import org.openrdf.sesame.query.rql.model.Intersection;

/**
 * A TableQueryResultListener that converts query results into an XML document.
 * The resulting XML document for a successful query has the following form:
 * <pre>
 * &lt;xml version="1.0" encoding="UTF-8"?&gt;
 * &lt;tableQueryResult&gt;
 *    &lt;header&gt;
 *        &lt;columnName&gt;X&lt;/columnName&gt;
 *        &lt;columnName&gt;Y&lt;/columnName&gt;
 *    &lt;/header&gt;
 *    &lt;tuple&gt;
 *        &lt;uri&gt;http://foo.com/bar&lt;/uri&gt;
 *        &lt;literal&gt;some text&lt;/literal&gt;
 *    &lt;/tuple&gt;
 *    &lt;tuple&gt;
 *        &lt;bNode&gt;node1234&lt;/bNode&gt;
 *        &lt;literal xml:lang="en"&gt;foobar&lt;/literal&gt;
 *    &lt;/tuple&gt;
 *    &lt;tuple&gt;
 *        &lt;bNode&gt;node5678&lt;/bNode&gt;
 *        &lt;literal datatype="http://www.w3.org/1999/02/22-rdf-syntax-ns#XMLLiteral"&gt;&lt;foo&gt;bar&lt;/foo&gt;&lt;/literal&gt;
 *    &lt;/tuple&gt;
 * &lt;/tableQueryResult&gt;
 * </pre>
 * In case of an error, the error will be reported in the query result
 * itself, e.g.:
 * <pre>
 * &lt;?xml version="1.0" encoding="UTF-8"?&gt;
 * &lt;tableQueryResult&gt;
 *    &lt;error type="MalformedQueryError"&gt;Expected '&gt;', found ' '&lt;/error&gt;
 * &lt;/tableQueryResult&gt;
 * </pre>
 * or:
 * <pre>
 * &lt;?xml version="1.0" encoding="UTF-8"?&gt;
 * &lt;tableQueryResult&gt;
 * &lt;header&gt;
 *    &lt;columnName&gt;X&lt;/columnName&gt;
 *    &lt;columnName&gt;Y&lt;/columnName&gt;
 * &lt;/header&gt;
 * &lt;tuple&gt;
 *    &lt;uri&gt;http://foo.com/bar&lt;/uri&gt;
 *    &lt;literal&gt;some text&lt;/literal&gt;
 * &lt;/tuple&gt;
 * &lt;error type="QueryEvaluationError"&gt;bladibla&lt;/error&gt;
 * &lt;/tableQueryResult&gt;
 * </pre>
 *
 * @author Peter van 't Hof
 * @author Arjohn Kampman
 */
public class XmlQueryResultWriter implements TableQueryResultListener {

/*----------+
| Variables |
+----------*/

	/**
	 * Flag indicating whether to write XML literals as plain XML.
	 */
	private boolean _plainXMLLiterals = false;

	/**
	 * XmlWriter to write XML to.
	 */
	private XmlWriter _xmlWriter;

/*-------------+
| Constructors |
+-------------*/

	/**
	 * Creates a new XmlQueryResultWriter (with plainXMLLiterals=false).
	 */
	public XmlQueryResultWriter(OutputStream out) {
		this(out, false);
	}

	public XmlQueryResultWriter(OutputStream out, boolean plainXMLLiterals) {
		_xmlWriter = new XmlWriter(out);
		_xmlWriter.setPrettyPrint(true);
		_plainXMLLiterals = plainXMLLiterals;
	}

	public XmlQueryResultWriter(Writer writer) {
		this(writer, false);
	}

	public XmlQueryResultWriter(Writer writer, boolean plainXMLLiterals) {
		_xmlWriter = new XmlWriter(writer);
		_xmlWriter.setPrettyPrint(true);
		_plainXMLLiterals = plainXMLLiterals;
	}
	
/*--------+
| Methods |
+--------*/

	/**
	 * Enables/disables addition of indentation characters and newlines in the
	 * XML document. By default, pretty-printing is set to <tt>true</tt>. If set
	 * to <tt>false</tt>, no indentation and newlines are added to the XML
	 * document. This method has to be used before writing starts (that is,
	 * before startTableQueryResult() is called).
	 */
	public void setPrettyPrint(boolean prettyPrint) {
		_xmlWriter.setPrettyPrint(prettyPrint);
	}
	
	// Implements TableQueryResultListener.startTableQueryResult()
	public void startTableQueryResult()
		throws IOException
	{
		startTableQueryResult(null);
	}

	// Implements TableQueryResultListener.startTableQueryResult(String[])
	public void startTableQueryResult(String[] columnHeaders) 
		throws IOException
	{
		_xmlWriter.startDocument();
		_xmlWriter.startTag("tableQueryResult");

		if (columnHeaders != null) {
			_xmlWriter.startTag("header");
			for (int i = 0; i < columnHeaders.length; i++) {
				_xmlWriter.textElement("columnName", columnHeaders[i]);
			}
			_xmlWriter.endTag("header");
		}
	}

	// Implements TableQueryResultListener.endTableQueryResult()
	public void endTableQueryResult()
		throws IOException
	{
		_xmlWriter.endTag("tableQueryResult");

		_xmlWriter.endDocument();
	}

	// Implements TableQueryResultListener.startTuple()
	public void startTuple()
		throws IOException
	{
		_xmlWriter.startTag("tuple");
	}

	// Implements TableQueryResultListener.endTuple()
	public void endTuple()
		throws IOException
	{
		_xmlWriter.endTag("tuple");
	}

	// Implements TableQueryResultListener.tupleValue(Value)
	public void tupleValue(Value value)
		throws IOException
	{
		if (value instanceof Intersection) {
			_writeIntersection((Intersection)value);
		}
		else if (value instanceof URI) {
			_writeURI((URI)value);
		}
		else if (value instanceof BNode) {
			_writeBNode((BNode)value);
		}
		else if (value instanceof Literal) {
			_writeLiteral((Literal)value);
		}
		else if (value == null) {
			_writeNull();
		}
	}

	// Implements TableQueryResultListener.tupleValue(Value)
	public void error(QueryErrorType errType, String msg)
		throws IOException
	{
		if (_xmlWriter == null) {
			// Error reported before startTableQueryResult() was called
			startTableQueryResult();
		}

		_xmlWriter.setAttribute("type", errType.toString());
		_xmlWriter.textElement("error", msg);
	}		

	private void _writeURI(URI uri)
		throws IOException
	{
		_xmlWriter.textElement("uri", uri.getURI());
	}

	private void _writeBNode(BNode bNode)
		throws IOException
	{
		_xmlWriter.textElement("bNode", bNode.getID());
	}

	private void _writeLiteral(Literal literal)
		throws IOException
	{
		boolean isXmlLiteral = false;

		if (literal.getLanguage() != null) {
			_xmlWriter.setAttribute("xml:lang", literal.getLanguage());
		}
		if (literal.getDatatype() != null) {
			URI datatype = literal.getDatatype();
			_xmlWriter.setAttribute("datatype", datatype.getURI());
			isXmlLiteral = datatype.equals(URIImpl.RDF_XMLLITERAL);
		}

		if (isXmlLiteral && _plainXMLLiterals) {
			_xmlWriter.unescapedTextElement("literal", literal.getLabel());
		}
		else {
			_xmlWriter.textElement("literal", literal.getLabel());
		}
	}

	private void _writeIntersection(Intersection intersection)
		throws IOException
	{
		_xmlWriter.startTag("intersection");
		
		Iterator i = intersection.getMembers().iterator();
		while (i.hasNext()) {
			Resource resource = (Resource)i.next();

			if (resource instanceof URI) {
				_writeURI((URI)resource);
			}
			else {
				_writeBNode((BNode)resource);
			}
		}
		
		_xmlWriter.endTag("intersection");
	}

	private void _writeNull()
		throws IOException
	{
		_xmlWriter.emptyElement("null");
	}
}
