/*  Sesame - Storage and Querying architecture for RDF and RDF Schema
 *  Copyright (C) 2001-2006 Aduna
 *
 *  Contact:
 *  	Aduna
 *  	Prinses Julianaplein 14 b
 *  	3817 CS Amersfoort
 *  	The Netherlands
 *  	tel. +33 (0)33 465 99 87
 *  	fax. +33 (0)33 465 99 87
 *
 *  	http://aduna-software.com/
 *  	http://www.openrdf.org/
 *
 *  This library is free software; you can redistribute it and/or
 *  modify it under the terms of the GNU Lesser General Public
 *  License as published by the Free Software Foundation; either
 *  version 2.1 of the License, or (at your option) any later version.
 *
 *  This library is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 *  Lesser General Public License for more details.
 *
 *  You should have received a copy of the GNU Lesser General Public
 *  License along with this library; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */
package org.openrdf.sesame.query;

import java.io.IOException;
import java.io.InputStream;
import java.util.ArrayList;
import java.util.List;
import java.util.Map;

import org.xml.sax.SAXException;
import org.xml.sax.XMLReader;

import org.openrdf.util.xml.SimpleSAXAdapter;
import org.openrdf.util.xml.SimpleSAXParser;

import org.openrdf.model.BNode;
import org.openrdf.model.Literal;
import org.openrdf.model.Resource;
import org.openrdf.model.URI;
import org.openrdf.model.ValueFactory;
import org.openrdf.model.impl.ValueFactoryImpl;

import org.openrdf.sesame.query.rql.model.Intersection;

/**
 * Reader for reading query results that are in Sesame's XML format for query
 * results. The reader assumes that the XML is wellformed and correct, and
 * does not do any extra verification. See the documentation for
 * <tt>XmlQueryResultWriter</tt> for a description of the XML format.
 *
 * @see XmlQueryResultWriter
 * @author Peter van 't Hof
 * @author Arjohn Kampman
 */
public class XmlQueryResultReader {

/*----------+
| Variables |
+----------*/

	/**
	 * The SimpleSAXParser that is used for parsing the XML.
	 **/
	private SimpleSAXParser _simpleSAXParser;

	/**
	 * The ValueFactory that is used to create URIs, BNodes and Literals.
	 **/
	private ValueFactory _valueFactory;

/*-------------+
| Constructors |
+-------------*/

	/**
	 * Creates a new XmlQueryResultReader that will use the supplied
	 * <tt>XMLReader</tt> for parsing the XML query result documents and an
	 * instance of <tt>ValueFactoryImpl</tt> to create any model objects.
	 */
	public XmlQueryResultReader(XMLReader xmlReader) {
		this(xmlReader, new ValueFactoryImpl());
	}

	/**
	 * Creates a new XmlQueryResultReader that will use the supplied
	 * <tt>XMLReader</tt> for parsing the XML query result documents and the
	 * supplied <tt>ValueFactory</tt> to create any model objects.
	 */
	public XmlQueryResultReader(XMLReader xmlReader, ValueFactory valueFactory) {
		_simpleSAXParser = new SimpleSAXParser(xmlReader);
		_valueFactory = valueFactory;
	}

/*--------+
| Methods |
+--------*/

	public synchronized void read(InputStream in, TableQueryResultListener listener)
		throws SAXException, IOException
	{
		_simpleSAXParser.setListener(new XmlQueryResultParser(listener));
		_simpleSAXParser.parse(in);
	}

/*---------------------------------+
| Inner class XmlQueryResultParser |
+---------------------------------*/

	class XmlQueryResultParser extends SimpleSAXAdapter {

	/*----------+
	| Variables |
	+----------*/

		/**
		 * TableQueryResultListener to report the query results to.
		 **/
		private TableQueryResultListener _listener;

		/**
		 * Flag indicating whether query result reporting has already started.
		 **/
		private boolean _queryResultStarted;

		/**
		 * Indicates if parsed values belong to an intersection.
		 **/
		private boolean _parsingIntersection = false;

		/**
		 * Members of an intersection.
		 **/
		private List _members = new ArrayList();

		/**
		 * The names of the columns.
		 **/
		private List _columnNames = new ArrayList();

	/*-------------+
	| Constructors |
	+-------------*/

		public XmlQueryResultParser(TableQueryResultListener listener) {
			_listener = listener;
		}

	/*----------+
	| Methods   |
	+----------*/

		public void startTag(String tagName, Map atts, String text)
			throws SAXException
		{
			try {
				if (tagName.equals("columnName")) {
					_columnNames.add(text);
				}
				else if (tagName.equals("tuple")) {
					if (!_queryResultStarted) {
						_reportQueryResultStart();
					}

					_listener.startTuple();
				}
				else if (tagName.equals("uri")) {
					URI uri = _valueFactory.createURI(text);

					if (_parsingIntersection) {
						// URI belongs to an intersection
						_members.add(uri);
					}
					else {
						_listener.tupleValue(uri);
					}
				}
				else if (tagName.equals("bNode")) {
					BNode bNode = _valueFactory.createBNode(text);

					if (_parsingIntersection) {
						// bNode belongs to an intersection
						_members.add(bNode);
					}
					else {
						_listener.tupleValue(bNode);
					}
				}
				else if (tagName.equals("literal")) {
					String xmlLang = (String)atts.get("xml:lang");
					String datatype = (String)atts.get("datatype");

					Literal literal = null;
					if (datatype != null) {
						URI dtURI = _valueFactory.createURI(datatype);
						literal = _valueFactory.createLiteral(text, dtURI);
					}
					else if (xmlLang != null) {
						literal = _valueFactory.createLiteral(text, xmlLang);
					}
					else {
						literal = _valueFactory.createLiteral(text);
					}

					_listener.tupleValue(literal);
				}
				else if (tagName.equals("null")) {
					_listener.tupleValue(null);
				}
				else if (tagName.equals("intersection")) {
					_parsingIntersection = true;
				}
				else if (tagName.equals("error")) {
					if (!_queryResultStarted) {
						_reportQueryResultStart();
					}

					String type = (String)atts.get("type");
					QueryErrorType errType = QueryErrorType.forValue(type);
					_listener.error(errType, text);
				}
			}
			catch (IOException e) {
				throw new SAXException(e);
			}
		}

		public void endTag(String tagName)
			throws SAXException
		{
			try {
				if (tagName.equals("tableQueryResult")) {
					if (!_queryResultStarted) {
						_reportQueryResultStart();
					}

					_listener.endTableQueryResult();
				}
				else if (tagName.equals("tuple")) {
					_listener.endTuple();
				}
				else if (tagName.equals("intersection")) {
					Intersection intersection = new Intersection();

					// Add all values from _members to intersection
					for (int i = 0; i < _members.size(); i++) {
						intersection.add( (Resource)_members.get(i) );
					}

					_parsingIntersection = false;
					_members.clear();

					_listener.tupleValue(intersection);
				}
			}
			catch (IOException e) {
				throw new SAXException(e);
			}
		}

		private void _reportQueryResultStart()
			throws IOException
		{
			if (_columnNames.isEmpty()) {
				_listener.startTableQueryResult();
			}
			else {
				String[] columns = new String[_columnNames.size()];
				for (int i = 0; i < _columnNames.size(); i++) {
					columns[i] = (String)_columnNames.get(i);
				}
				_listener.startTableQueryResult(columns);
			}

			_queryResultStarted = true;
		}
	}
}
