/*  Sesame - Storage and Querying architecture for RDF and RDF Schema
 *  Copyright (C) 2001-2006 Aduna
 *
 *  Contact: 
 *  	Aduna
 *  	Prinses Julianaplein 14 b
 *  	3817 CS Amersfoort
 *  	The Netherlands
 *  	tel. +33 (0)33 465 99 87
 *  	fax. +33 (0)33 465 99 87
 *
 *  	http://aduna-software.com/
 *  	http://www.openrdf.org/
 *  
 *  This library is free software; you can redistribute it and/or
 *  modify it under the terms of the GNU Lesser General Public
 *  License as published by the Free Software Foundation; either
 *  version 2.1 of the License, or (at your option) any later version.
 *
 *  This library is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 *  Lesser General Public License for more details.
 *
 *  You should have received a copy of the GNU Lesser General Public
 *  License along with this library; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */
package org.openrdf.sesame.query;

import java.io.IOException;
import java.io.OutputStream;

import org.openrdf.util.xml.XmlWriter;

import org.openrdf.model.BNode;
import org.openrdf.model.Literal;
import org.openrdf.model.URI;
import org.openrdf.model.Value;

import org.openrdf.sesame.constants.RDFFormat;

/**
 * A TableQueryResultListener that converts query results into an RDF
 * document containing variable bindings. This format is based on the
 * work done by Andy Seaborne. See <a
 * href="http://www.w3.org/2003/03/rdfqr-tests/recording-query-results.html">
 * http://www.w3.org/2003/03/rdfqr-tests/recording-query-results.html</a>
 * for a description of the results format.
 *
 * @author Arjohn Kampman
 * @version $Revision: 1.9.4.2 $
 */
public class VarBindingRdfWriter implements TableQueryResultListener {

/*----------+
| Variables |
+----------*/

	private RDFFormat _rdfFormat;
	private OutputStream _out;

	/** XmlWriter to write XML to. */
	private XmlWriter _xmlWriter;

	private String[] _varNames;
	private int _currentColumn;

/*-------------+
| Constructors |
+-------------*/

	/**
	 * Creates a new VarBindingRdfWriter.
	 */
	public VarBindingRdfWriter(OutputStream out) {
		this(RDFFormat.RDFXML, out);
	}

	public VarBindingRdfWriter(RDFFormat rdfFormat, OutputStream out) {
		// FIXME: handle N-Triples and N3 encodings
		_rdfFormat = rdfFormat;
		_out = out;
	}

/*--------+
| Methods |
+--------*/

	public void startTableQueryResult()
		throws IOException
	{
		throw new IOException("This output format requires column header reporting");
	}

	public void startTableQueryResult(String[] columnHeaders)
		throws IOException
	{
		_xmlWriter = new XmlWriter(_out);
		_xmlWriter.setPrettyPrint(true);

		// Writer header
		_xmlWriter.startDocument();
		_xmlWriter.setAttribute("xmlns:rdf", "http://www.w3.org/1999/02/22-rdf-syntax-ns#");
		_xmlWriter.setAttribute("xmlns:rs", "http://jena.hpl.hp.com/2003/03/result-set#");
		_xmlWriter.startTag("rdf:RDF");

		// Write experimantal format comment
		_xmlWriter.emptyLine();
		_xmlWriter.comment(
				"  This is an experimental results format. See\n" +
				"  http://www.w3.org/2003/03/rdfqr-tests/recording-query-results.html\n" +
				"  for more info about this format.");
		_xmlWriter.emptyLine();

		_xmlWriter.setAttribute("rdf:about", "");
		_xmlWriter.startTag("rs:ResultSet");

		// Copy and write 'var names'
		_varNames = new String[columnHeaders.length];

		for (int i = 0; i < columnHeaders.length; i++) {
			_varNames[i] = columnHeaders[i];
			_xmlWriter.textElement("rs:resultVariable", _varNames[i]);
		}
	}

	public void endTableQueryResult()
		throws IOException
	{
		_xmlWriter.endTag("rs:ResultSet");
		_xmlWriter.endTag("rdf:RDF");

		_xmlWriter.endDocument();
		_out.close();
	}


	public void startTuple()
		throws IOException
	{
		_xmlWriter.startTag("rs:solution");
		_xmlWriter.startTag("rs:ResultSolution");

		_currentColumn = 0;
	}

	public void endTuple()
		throws IOException
	{
		_xmlWriter.endTag("rs:ResultSolution");
		_xmlWriter.endTag("rs:solution");
	}

	public void tupleValue(Value value)
		throws IOException
	{
		_xmlWriter.setAttribute("rdf:parseType", "Resource");
		_xmlWriter.startTag("rs:binding");
		_xmlWriter.textElement("rs:variable", _varNames[_currentColumn]);

		if (value instanceof URI) {
			_writeURI((URI)value);
		}
		else if (value instanceof BNode) {
			_writeBNode((BNode)value);
		}
		else if (value instanceof Literal) {
			_writeLiteral((Literal)value);
		}
		else if (value == null) {
			_writeNull();
		}

		_xmlWriter.endTag("rs:binding");

		_currentColumn++;
	}

	private void _writeURI(URI uri)
		throws IOException
	{
		_xmlWriter.setAttribute("rdf:resource", uri.getURI());
		_xmlWriter.emptyElement("rs:value");
	}

	private void _writeBNode(BNode bNode)
		throws IOException
	{
		_xmlWriter.setAttribute("rdf:nodeID", bNode.getID());
		_xmlWriter.emptyElement("rs:value");
	}

	private void _writeLiteral(Literal literal)
		throws IOException
	{
		if (literal.getLanguage() != null) {
			_xmlWriter.setAttribute("xml:lang", literal.getLanguage());
		}

		if (literal.getDatatype() != null) {
			_xmlWriter.setAttribute("rdf:datatype", literal.getDatatype().getURI());
		}

		_xmlWriter.textElement("rs:value", literal.getLabel());
	}

	private void _writeNull()
		throws IOException
	{
		_xmlWriter.setAttribute("rdf:resource", "http://jena.hpl.hp.com/2003/03/result-set#undef");
		_xmlWriter.emptyElement("rs:value");
	}

	public void error(QueryErrorType errType, String msg)
		throws IOException
	{
		if (_xmlWriter == null) {
			// Error reported before startTableQueryResult was called.
			startTableQueryResult(new String[] {});
		}

		_xmlWriter.startTag("error");
		_xmlWriter.textElement("msg", msg);
		_xmlWriter.endTag("error");
	}
}
