/*  Sesame - Storage and Querying architecture for RDF and RDF Schema
 *  Copyright (C) 2001-2006 Aduna
 *
 *  Contact: 
 *  	Aduna
 *  	Prinses Julianaplein 14 b
 *  	3817 CS Amersfoort
 *  	The Netherlands
 *  	tel. +33 (0)33 465 99 87
 *  	fax. +33 (0)33 465 99 87
 *
 *  	http://aduna-software.com/
 *  	http://www.openrdf.org/
 *  
 *  This library is free software; you can redistribute it and/or
 *  modify it under the terms of the GNU Lesser General Public
 *  License as published by the Free Software Foundation; either
 *  version 2.1 of the License, or (at your option) any later version.
 *
 *  This library is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 *  Lesser General Public License for more details.
 *
 *  You should have received a copy of the GNU Lesser General Public
 *  License along with this library; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

package org.openrdf.sesame.query;

import java.io.IOException;

import org.openrdf.model.Value;

/**
 * <p>
 * A listener for table-like query results. The results are returned row-by-row.
 * The values of the rows (tuples) are returned one after the other through
 * the method tupleValue(Value). So a query result like:
 * </p>
 * <table border="1">
 *   <tr><td>A</td><td>B</td><td>C</td></tr>
 *   <tr><td>X</td><td>Y</td><td>Z</td></tr>
 * </table>
 *
 * <p>
 * will result in the following calls to this listener:
 * </p>
 * <pre>
 * startTableQueryResult()
 *     startTuple()
 *         tupleValue(A)
 *         tupleValue(B)
 *         tupleValue(C)
 *     endTuple()
 *     startTuple()
 *         tupleValue(X)
 *         tupleValue(Y)
 *         tupleValue(Z)
 *     endTuple()
 * endTableQueryResult()
 * </pre>
 *
 * <p>
 * Note1: only one of the <tt>startTableQueryResult()</tt> methods will be called.<br>
 * Note2: in case of an error occuring during query evaluation, an error message
 * will be reported using the <tt>reportError()</tt> method.
 * </p>
 **/
public interface TableQueryResultListener {

/*------------------+
| Methods           |
+------------------*/

	/**
	 * Indicates the start of a table-like query result.
	 **/
	public void startTableQueryResult()
		throws IOException;

	/**
	 * Indicates the start of a table-like query result. The supplied columnHeaders
	 * are an indication of the values that are in a specific column. For example,
	 * a query <tt>select A, B from * ....</tt> could have column headers
	 * <tt>A</tt> and <tt>B</tt>.
	 **/
	public void startTableQueryResult(String[] columnHeaders)
		throws IOException;
	
	/**
	 * Indicates the end of a table-like query result.
	 **/
	public void endTableQueryResult()
		throws IOException;

	/**
	 * Indicates the start of a tuple/row.
	 **/
	public void startTuple()
		throws IOException;

	/**
	 * Indicates the end of a tuple/row.
	 **/
	public void endTuple()
		throws IOException;

	/**
	 * Delivers the next value in the current tuple/row.
	 *
	 * @param value The next value in the current tuple/row.
	 **/
	public void tupleValue(Value value)
		throws IOException;

	/**
	 * Reports an error that has occurred during the query evaluation.
	 *
	 * @param errType The type of error.
	 * @param msg A message describing the error.
	 **/
	public void error(QueryErrorType errType, String msg)
		throws IOException;
}
