/*  Sesame - Storage and Querying architecture for RDF and RDF Schema
 *  Copyright (C) 2001-2006 Aduna
 *
 *  Contact: 
 *  	Aduna
 *  	Prinses Julianaplein 14 b
 *  	3817 CS Amersfoort
 *  	The Netherlands
 *  	tel. +33 (0)33 465 99 87
 *  	fax. +33 (0)33 465 99 87
 *
 *  	http://aduna-software.com/
 *  	http://www.openrdf.org/
 *  
 *  This library is free software; you can redistribute it and/or
 *  modify it under the terms of the GNU Lesser General Public
 *  License as published by the Free Software Foundation; either
 *  version 2.1 of the License, or (at your option) any later version.
 *
 *  This library is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 *  Lesser General Public License for more details.
 *
 *  You should have received a copy of the GNU Lesser General Public
 *  License along with this library; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

package org.openrdf.sesame.query;

import java.io.IOException;
import java.util.HashSet;
import java.util.Iterator;
import java.util.Set;

import org.openrdf.model.Value;

/**
 * A TableQueryResultListener that filters any duplicates from the query
 * results and forwards these filtered results to another
 * TableQueryResultListener.
 **/
public class TableQueryDuplicatesFilter implements TableQueryResultListener {

	/** Recorded query result. */
	protected Set _queryResult;

	/** The tuple that is currently being processed. **/
	protected Tuple _tuple;

	/** The listener to forward the filtered query results to. **/
	protected TableQueryResultListener _listener;

	/**
	 * Creates a new TableQueryDuplicatesFilter.
	 * @param l The TableQueryResultListener that will receive the
	 * filtered query results.
	 **/
	public TableQueryDuplicatesFilter(TableQueryResultListener l) {
		_listener = l;
		_queryResult = new HashSet();
	}

/*------------------------------------------------+
| Methods from interface TableQueryResultListener |
+------------------------------------------------*/

	public void startTableQueryResult()
		throws IOException
	{
		_listener.startTableQueryResult();
	}

	public void startTableQueryResult(String[] columnHeaders)
		throws IOException
	{
		_listener.startTableQueryResult(columnHeaders);
	}

	public void endTableQueryResult()
		throws IOException
	{
		_listener.endTableQueryResult();
		_queryResult.clear();
	}

	public void startTuple()
		throws IOException
	{
		_tuple = new Tuple();
	}

	public void tupleValue(Value value)
		throws IOException
	{
		_tuple.add(value);
	}

	public void endTuple()
		throws IOException
	{
		if (!_queryResult.contains(_tuple)) {
			// new unique result
			_queryResult.add(_tuple);

			// forward the result
			_listener.startTuple();

			Iterator i = _tuple.iterator();
			while (i.hasNext()) {
				_listener.tupleValue((Value)i.next());
			}			

			_listener.endTuple();
		}
	}

	public void error(QueryErrorType errType, String msg)
		throws IOException
	{
		_listener.error(errType, msg);
	}
}
