/*  Sesame - Storage and Querying architecture for RDF and RDF Schema
 *  Copyright (C) 2001-2006 Aduna
 *
 *  Contact: 
 *  	Aduna
 *  	Prinses Julianaplein 14 b
 *  	3817 CS Amersfoort
 *  	The Netherlands
 *  	tel. +33 (0)33 465 99 87
 *  	fax. +33 (0)33 465 99 87
 *
 *  	http://aduna-software.com/
 *  	http://www.openrdf.org/
 *  
 *  This library is free software; you can redistribute it and/or
 *  modify it under the terms of the GNU Lesser General Public
 *  License as published by the Free Software Foundation; either
 *  version 2.1 of the License, or (at your option) any later version.
 *
 *  This library is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 *  Lesser General Public License for more details.
 *
 *  You should have received a copy of the GNU Lesser General Public
 *  License along with this library; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

package org.openrdf.sesame.query;

import java.io.IOException;
import java.io.OutputStream;

import org.openrdf.model.Resource;
import org.openrdf.model.URI;
import org.openrdf.model.Value;

import org.openrdf.rio.RdfDocumentWriter;
import org.openrdf.rio.n3.N3Writer;
import org.openrdf.rio.ntriples.NTriplesWriter;
import org.openrdf.rio.rdfxml.RdfXmlWriter;
import org.openrdf.rio.turtle.TurtleWriter;

import org.openrdf.sesame.constants.RDFFormat;

public class RdfGraphWriter implements GraphQueryResultListener {

/*----------------------------------------------+
| Variables                                     |
+----------------------------------------------*/

	protected OutputStream _out;
	protected RdfDocumentWriter _rdfDocWriter;

/*----------------------------------------------+
| Constructors                                  |
+----------------------------------------------*/

	/**
	 * Creates a new RdfGraphWriter that will write the received query
	 * results to the supplied <tt>OutputStream</tt>. The <tt>rdfFormat</tt>
	 * parameter controls the serialization format of the written RDF
	 *
	 * @param out The OutputStream to write the RDF to.
	 * @param rdfFormat One of the constants defined in class <tt>RDFFormat</tt>.
	 **/
	public RdfGraphWriter(RDFFormat rdfFormat, OutputStream out)
		throws IOException
	{
		_out = out;

		if (rdfFormat == RDFFormat.RDFXML) {
			_rdfDocWriter = new RdfXmlWriter(out);
		}
		else if (rdfFormat == RDFFormat.NTRIPLES) {
			_rdfDocWriter = new NTriplesWriter(out);
		}
		else if (rdfFormat == RDFFormat.N3) {
			_rdfDocWriter = new N3Writer(out);
		}
		else if (rdfFormat == RDFFormat.TURTLE) {
			_rdfDocWriter = new TurtleWriter(out);
		}
		else {
			throw new IllegalArgumentException(
					"Unsupported value for rdfFormat: " + rdfFormat.toString());
		}
	}

	/** Indicates the start of a graph-like query result. **/
	public void startGraphQueryResult()
		throws IOException
	{
		_rdfDocWriter.startDocument();
	}

	/** Indicates the end of a graph-like query result. **/
	public void endGraphQueryResult()
		throws IOException
	{
		_rdfDocWriter.endDocument();

		_out.flush();
		_out.close();
	}

	public void namespace(String prefix, String name)
		throws IOException
	{
		_rdfDocWriter.setNamespace(prefix, name);
	}

	public void triple(Resource subj, URI pred, Value obj)
		throws IOException
	{
		_rdfDocWriter.writeStatement(subj, pred, obj);
	}

	public void reportError(String msg)
		throws IOException
	{
		_rdfDocWriter.writeComment("ERROR: " + msg);
	}
}
