/*  Sesame - Storage and Querying architecture for RDF and RDF Schema
 *  Copyright (C) 2001-2006 Aduna
 *
 *  Contact: 
 *  	Aduna
 *  	Prinses Julianaplein 14 b
 *  	3817 CS Amersfoort
 *  	The Netherlands
 *  	tel. +33 (0)33 465 99 87
 *  	fax. +33 (0)33 465 99 87
 *
 *  	http://aduna-software.com/
 *  	http://www.openrdf.org/
 *  
 *  This library is free software; you can redistribute it and/or
 *  modify it under the terms of the GNU Lesser General Public
 *  License as published by the Free Software Foundation; either
 *  version 2.1 of the License, or (at your option) any later version.
 *
 *  This library is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 *  Lesser General Public License for more details.
 *
 *  You should have received a copy of the GNU Lesser General Public
 *  License along with this library; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

package org.openrdf.sesame.query;

import java.io.IOException;
import java.util.ArrayList;
import java.util.List;

import org.openrdf.model.Value;

/**
 * A TableQueryResultListener that can be used to build a QueryResultsTable.
 *
 * @see QueryResultsTable
 **/
public class QueryResultsTableBuilder implements TableQueryResultListener {

/*-----------------------------------------------------+
| Variables                                            |
+-----------------------------------------------------*/

	/**
	 * The QueryResultsTable.
	 **/
	private QueryResultsTable _queryResultsTable;

	/**
	 * The tuple that is currently being received.
	 **/
	private List _currentTuple;

	private String[] _columnNames;
	
/*-----------------------------------------------------+
| Constructors                                         |
+-----------------------------------------------------*/

	/**
	 * Creates a new QueryResultsTableBuilder.
	 **/
	public QueryResultsTableBuilder() {
	}

/*-----------------------------------------------------+
| Methods                                              |
+-----------------------------------------------------*/

	/**
	 * Gets the QueryResultsTable that has been built. This method only
	 * returns a table when the results have (partially) been parsed, and
	 * when the builder hasn't been cleared in the meantime. The table can 
	 * be empty.
	 *
	 * @return A QueryResultsTable, or null if not available (this can happen 
	 * when invoked before the query is evaluated). Note that an
	 * empty QueryResultsTable will be returned if the query had no results.
	 **/
	public QueryResultsTable getQueryResultsTable() {
		return _queryResultsTable;
	}

	/**
	 * Clear the builder. Any table that has been built so far will be
	 * removed.
	 **/
	public void clear() {
		// remove any results
		_queryResultsTable = null;
	}

/*-----------------------------------------------------+
| Methods from the TableQueryResultListener interface       |
+-----------------------------------------------------*/

	/**
	 * Method needed by the TableQueryResultListener interface.
	 **/
	public void startTableQueryResult() {
		// remove any previous results
		_queryResultsTable = null;

		_currentTuple = new ArrayList();
	}

	public void startTableQueryResult(String[] columnHeaders) {
		_columnNames = columnHeaders;
		startTableQueryResult();
	}


	/**
	 * Method needed by the TableQueryResultListener interface.
	 **/
	public void endTableQueryResult() {
		_currentTuple = null;

		// In case the query has no results, no table has been created yet.
		if (_queryResultsTable == null) {
			_queryResultsTable = new QueryResultsTable(0, _columnNames);
		}
	}

	/**
	 * Method needed by the TableQueryResultListener interface.
	 **/
	public void startTuple() {
	}

	/**
	 * Method needed by the TableQueryResultListener interface.
	 **/
	public void endTuple() {
		if (_queryResultsTable == null) {
			// First row, create a table for the results
			_queryResultsTable =
				new QueryResultsTable(_currentTuple.size(), _columnNames);
		}

		_queryResultsTable._addRow(_currentTuple);
		_currentTuple.clear();
	}

	/**
	 * Method needed by the TableQueryResultListener interface.
	 **/
	public void tupleValue(Value value) {
		_currentTuple.add(value);
	}
	
	public void error(QueryErrorType errType, String msg)
		throws IOException
	{
		// FIXME do nothing
		//throw new QueryEvaluationException(msg);
	}
}
