/*  Sesame - Storage and Querying architecture for RDF and RDF Schema
 *  Copyright (C) 2001-2006 Aduna
 *
 *  Contact:
 *  	Aduna
 *  	Prinses Julianaplein 14 b
 *  	3817 CS Amersfoort
 *  	The Netherlands
 *  	tel. +33 (0)33 465 99 87
 *  	fax. +33 (0)33 465 99 87
 *
 *  	http://aduna-software.com/
 *  	http://www.openrdf.org/
 *
 *  This library is free software; you can redistribute it and/or
 *  modify it under the terms of the GNU Lesser General Public
 *  License as published by the Free Software Foundation; either
 *  version 2.1 of the License, or (at your option) any later version.
 *
 *  This library is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 *  Lesser General Public License for more details.
 *
 *  You should have received a copy of the GNU Lesser General Public
 *  License along with this library; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

package org.openrdf.sesame.query;

import java.util.ArrayList;
import java.util.List;

import org.openrdf.model.Value;

import java.io.Serializable;

/**
 * A table for query results. This table is provided for those who
 * do not want to use the streaming approach provided by the
 * <tt>QueryResultListener</tt>. The <tt>QueryResultsTableBuilder</tt> can be used to
 * build the tables. The query results table is guaranteed to be rectangular
 * (i.e. the number of columns is equal for all rows).
 *
 * @see TableQueryResultListener
 * @see QueryResultsTableBuilder
 **/
public class QueryResultsTable implements Serializable {

/*----------+
| Variables |
+----------*/

	/**
	 * The number of columns of this table.
	 **/
	private int _colCount = 0;

	/**
	 * List containing all rows. The rows are again of type List and
	 * contain the values for each column in that row.
	 **/
	private List _rowList = new ArrayList();

	/**
	 * Array of strings with column names.
	 */
	private String[] _columnNames;

/*-------------+
| Constructors |
+-------------*/

	/**
	 * Creates a new QueryResultsTable that can store a query result with the
	 * indicated number of columns.
	 *
	 * @param colCount The number of columns that the query result has.
	 **/
	public QueryResultsTable(int colCount) {
		_colCount = colCount;
	}

	public QueryResultsTable(int colCount, String[] columnNames) {
		this(colCount);
		_columnNames = columnNames;
	}

/*--------+
| Methods |
+--------*/

	/**
	 * Gets the number of rows in this table.
	 **/
	public int getRowCount() {
		return _rowList.size();
	}

	/**
	 * Gets the number of columns in this table.
	 **/
	public int getColumnCount() {
		return _colCount;
	}

	/**
	 * return the names of all columns in an array.
	 *
	 * @return an array containing the names of all columns, or null if not known.
	 */
	public String[] getColumnNames() {
		return _columnNames;
	}

	/**
	 * Return the name of the column at index 'column'
	 *
	 * @param column the column index
	 * @return the name of the column, or null if not known.
	 */
	public String getColumnName(int column) {
		if (_columnNames == null) {
			return null;
		}
		return _columnNames[column];
	}

	/**
	 * Fetches the value at the specified row and column in this table. Both
	 * row and column are 0-based.
	 *
	 * @exception IndexOutOfBoundsException If the requested row or column
	 * is out of the range of this table.
	 **/
	public Value getValue(int row, int column) {
		List columns = (List)_rowList.get(row);
		return (Value)columns.get(column);
	}

	/**
	 * Adds a row to this table. The number of values in the row should be
	 * equal to the number of columns of this table. Further, all objects in
	 * the list should be of type <tt>org.openrdf.model.Value</tt>,
	 * but this is not enforced. Adding objects of other types will lead to
	 * <tt>ClassCastException</tt>s when the object is fetched from the table.
	 *
	 * @param values A List of Value objects.
	 * @exception IllegalArgumentException If the number of values in the row
	 * is not equal to the number of columns of this table.
	 **/
	void _addRow(List values) {
		if (values.size() != _colCount) {
			throw new IllegalArgumentException(
				"Number of values (" + values.size() +
				") in the row is not equal to the number of columns (" +
				_colCount + ") of this table");
		}

		_rowList.add( new ArrayList(values) );
	}

	/**
	 * Compares this QueryResultsTable with another, returning <tt>true</tt> if
	 * both contain the same values. The column names do not necessarily need to
	 * be identical for the tables to be considered equal.
	 **/
	public boolean equals(Object other) {
		if (other instanceof QueryResultsTable) {
			QueryResultsTable otherTable = (QueryResultsTable)other;

			return
				_colCount == otherTable._colCount &&
				_rowList.equals(otherTable._rowList);
		}

		return false;
	}

	// Overrides Object.hashCode()
	public int hashCode() {
		if (_rowList.isEmpty()) {
			return 0;
		}
		else {
			// use hash code of the middle row
			List columns = (List)_rowList.get( _rowList.size() / 2 );
			return columns.hashCode();
		}
	}

	// Overrides Object.toString()
	public String toString() {
		StringBuffer buf = new StringBuffer(20 * _rowList.size() * _colCount + 10);

		if (_columnNames != null) {
			for (int i = 0; i < _columnNames.length; i++) {
				if (i > 0) {
					buf.append("\t| ");
				}
				buf.append(_columnNames[i]);
			}
			buf.append('\n');

			int dashCount = buf.length() + 7*(_columnNames.length-1);
			for (int i = 0; i < dashCount; i++) {
				buf.append('-');
			}
			buf.append('\n');
		}

		for (int i = 0; i < _rowList.size(); i++) {
			List columns = (List)_rowList.get(i);

			for (int j = 0; j < columns.size(); j++) {
				if (j > 0) {
					buf.append("\t| ");
				}
				buf.append( columns.get(j).toString() );
			}
			buf.append('\n');
		}

		return buf.toString();
	}
}
