/*  Sesame - Storage and Querying architecture for RDF and RDF Schema
 *  Copyright (C) 2001-2006 Aduna
 *
 *  Contact:
 *  	Aduna
 *  	Prinses Julianaplein 14 b
 *  	3817 CS Amersfoort
 *  	The Netherlands
 *  	tel. +33 (0)33 465 99 87
 *  	fax. +33 (0)33 465 99 87
 *
 *  	http://aduna-software.com/
 *  	http://www.openrdf.org/
 *
 *  This library is free software; you can redistribute it and/or
 *  modify it under the terms of the GNU Lesser General Public
 *  License as published by the Free Software Foundation; either
 *  version 2.1 of the License, or (at your option) any later version.
 *
 *  This library is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 *  Lesser General Public License for more details.
 *
 *  You should have received a copy of the GNU Lesser General Public
 *  License along with this library; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */
package org.openrdf.sesame.query;

import java.io.IOException;
import java.util.HashMap;
import java.util.Iterator;
import java.util.Map;

import org.openrdf.model.Graph;
import org.openrdf.model.Resource;
import org.openrdf.model.URI;
import org.openrdf.model.Value;

import org.openrdf.sesame.Sesame;
import org.openrdf.sesame.config.AccessDeniedException;
import org.openrdf.sesame.config.ConfigurationException;
import org.openrdf.sesame.config.RepositoryConfig;
import org.openrdf.sesame.repository.local.LocalRepository;
import org.openrdf.sesame.sail.RdfRepository;
import org.openrdf.sesame.sail.SailUpdateException;

/**
 * An implementation of <tt>GraphQueryResultListener</tt> that can be used to
 * create a Graph from a query result.
 * 
 * @author Jeen Broekstra
 * @author Arjohn Kampman
 * @version $Revision: 1.5.2.2 $
 */
public class QueryResultsGraphBuilder implements GraphQueryResultListener {

/*----------+
| Variables |
+----------*/

	private LocalRepository _repository;
	private RdfRepository _sail;
	private Map _namespacesTable;

/*-------------+
| Constructors |
+-------------*/

	/**
	 * Creates a new QueryResultsGraphBuilder.
	 **/
	public QueryResultsGraphBuilder() {
		_namespacesTable = new HashMap(16);
	}

/*--------+
| Methods |
+--------*/

	/**
	 * Gets the graph that has been build from the query result.
	 * 
	 * @return A <tt>Graph</tt> object, or <tt>null</tt> if the query result has
	 * not yet been processed.
	 */
	public Graph getGraph() {
		try {
			return _repository.getGraph();
		}
		catch (AccessDeniedException e) {
			// Repository was configured to be world-readable and -writeable
			throw new RuntimeException(e);
		}
	}

	// implements GraphQueryResultListener.startGraphQueryResult()
	public void startGraphQueryResult()
		throws IOException
	{
		try {
			RepositoryConfig config = new RepositoryConfig("", true, true);
			config.stackSail( new org.openrdf.sesame.sailimpl.memory.RdfRepositoryConfig() );

			_repository = Sesame.getService().createRepository(config);

			_sail = (RdfRepository)_repository.getSail();
			_sail.startTransaction();
		}
		catch (ConfigurationException e) {
			IOException ioe = new IOException("Unable to create memory repository");
			ioe.initCause(e);
			throw ioe;
		}
	}

	// implements GraphQueryResultListener.endGraphQueryResult()
	public void endGraphQueryResult() {
		Iterator iter = _namespacesTable.entrySet().iterator();
		while (iter.hasNext()) {
			Map.Entry entry = (Map.Entry)iter.next();

			String namespace = (String)entry.getKey();
			String prefix = (String)entry.getValue();

			try {
				_sail.changeNamespacePrefix(namespace, prefix);
			}
			catch (SailUpdateException e) {
				// ignore
			}
		}

		_namespacesTable.clear();

		_sail.commitTransaction();
	}

	// implements GraphQueryResultListener.namespace(String,String)
	public void namespace(String prefix, String name) {
		_namespacesTable.put(name, prefix);
	}

	// implements GraphQueryResultListener.triple(...)
	public void triple(Resource subj, URI pred, Value obj)
		throws IOException
	{
		try {
			_sail.addStatement(subj, pred, obj);
		}
		catch (SailUpdateException e) {
			IOException ioe = new IOException("Unable to add statement to memory sail");
			ioe.initCause(e);
			throw ioe;
		}
	}

	// implements GraphQueryResultListener.reportError(String)
	public void reportError(String msg)
		throws IOException
	{
		//FIXME do nothing
		//throw new QueryEvaluationException(msg);
	}
}
