/*  Sesame - Storage and Querying architecture for RDF and RDF Schema
 *  Copyright (C) 2001-2006 Aduna
 *
 *  Contact: 
 *  	Aduna
 *  	Prinses Julianaplein 14 b
 *  	3817 CS Amersfoort
 *  	The Netherlands
 *  	tel. +33 (0)33 465 99 87
 *  	fax. +33 (0)33 465 99 87
 *
 *  	http://aduna-software.com/
 *  	http://www.openrdf.org/
 *  
 *  This library is free software; you can redistribute it and/or
 *  modify it under the terms of the GNU Lesser General Public
 *  License as published by the Free Software Foundation; either
 *  version 2.1 of the License, or (at your option) any later version.
 *
 *  This library is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 *  Lesser General Public License for more details.
 *
 *  You should have received a copy of the GNU Lesser General Public
 *  License along with this library; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

package org.openrdf.sesame.query;

import java.io.IOException;
import java.io.OutputStream;
import java.io.OutputStreamWriter;
import java.io.Writer;
import java.util.Iterator;

import org.openrdf.util.xml.XmlUtil;

import org.openrdf.model.Resource;
import org.openrdf.model.Value;

import org.openrdf.sesame.query.rql.model.Intersection;
import org.openrdf.sesame.server.http.RdfExplorerUtil;

/**
 * A TableQueryResultListener that converts the query results into an HTML document.
 **/
public class HtmlTableWriter implements TableQueryResultListener {

/*----------------------------------------------+
| Variables                                     |
+----------------------------------------------*/

	protected Writer _out;
	protected String _repository;
	protected String _resultId;
	protected String _styleSheetUrl;

	protected int _nrOfResults;
	protected int _nrOfColumns;
	protected long _startTime;
	protected boolean _tupleOpen, _tableOpen;

/*----------------------------------------------+
| Constructors                                  |
+----------------------------------------------*/

	public HtmlTableWriter(OutputStream out, String repository, String styleSheetUrl)
		throws IOException
	{
		this(out, repository, styleSheetUrl, "query_result");
	}

	public HtmlTableWriter(OutputStream out, String repository, String styleSheetUrl, String resultId)
		throws IOException
	{
		_out = new OutputStreamWriter(out, "UTF-8");
		_repository = repository;
		_resultId = resultId;

		_styleSheetUrl = styleSheetUrl;
		_tupleOpen = false;
		_tableOpen = false;		

		// XXX: this is a bit of a hack, but want to include parsing and
		// optimization time in the evaluation time.
		_startTime = System.currentTimeMillis();
	}


/*----------------------------------------------+
| Methods                                       |
+----------------------------------------------*/

	public void startTableQueryResult()
		throws IOException
	{
		_nrOfResults = 0;

		_out.write("<html>\n");
		_out.write("<head>\n");

		if (_styleSheetUrl != null) {
			_out.write("<link rel=\"stylesheet\" type=\"text/css\" href=\"" + _styleSheetUrl + "\">\n");
		}

		_out.write("<meta http-equiv=\"Content-Type\" content=\"text/html; charset=utf-8\">");

		_out.write("<title>Query Results</title>\n");
		_out.write("</head>\n");
		_out.write("<body bgcolor=\"white\">\n");
		_out.write("<table>\n");
		_out.write("<tr><th class=\"nobg\"");
		if (_nrOfColumns > 0) {
			_out.write(" colspan=\"" + _nrOfColumns + "\"");
		}
		_out.write(">Query results:</th></tr>\n");
		_tableOpen = true;
	}

	public void startTableQueryResult(String[] columnHeaders) 
		throws IOException 
	{
		_nrOfColumns = columnHeaders.length;

		startTableQueryResult();

		_out.write("  <tr>\n");
		for (int i = 0; i < _nrOfColumns; i++) {
			_out.write("    <th>");
			_out.write( XmlUtil.escapeCharacterData(columnHeaders[i]) );
			_out.write("</th>\n");
		}
		_out.write("  </tr>\n");
	}
		
	public void endTableQueryResult()
		throws IOException
	{
		long evalTime = System.currentTimeMillis() - _startTime;

		if (_tupleOpen) {
			endTuple();
		}
		if (_tableOpen) {
			_out.write("<tr><td class=\"nobg\"");
			if (_nrOfColumns > 0) {
				_out.write(" colspan=\"" + _nrOfColumns + "\"");
			}
			_out.write(">");
			_out.write(_nrOfResults + " results found in " + evalTime + " ms.</td></tr>\n");
			endTable();
		}
		else {
			_out.write("<p>" + _nrOfResults + " results found in " + evalTime + " ms.</p>\n");
		}
		_out.write("</body>\n");
		_out.write("</html>\n");

		_out.flush();
		_out.close();
	}


	public void startTuple()
		throws IOException
	{
		_nrOfResults++;
		_out.write("  <tr>\n");
		_tupleOpen = true;
	}

	public void endTuple()
		throws IOException
	{
		_out.write("  </tr>\n");
		_tupleOpen = false;
	}

	public void tupleValue(Value value)
		throws IOException
	{
		_out.write("      <td>");

		if (value == null) {
			_out.write("NULL");
		}
		else if (value instanceof Intersection) {
			Intersection intersect = (Intersection)value;
			Iterator iter = intersect.getMembers().iterator();

			if (iter.hasNext()) {
				_out.write("{&nbsp;\n");
				while (iter.hasNext()) {
					_writeValueLink( (Resource)iter.next() );

					if (iter.hasNext()) {
						_out.write(",<br>&nbsp;&nbsp;\n");
					}
				}
				_out.write("&nbsp;}\n");
			}
			else {
				_out.write("{}\n");
			}
		}
		else {
			_writeValueLink(value);
		}

		_out.write("</td>\n");
	}

	private void _writeValueLink(Value value)
		throws IOException
	{
		_out.write("<a target=\"rdfexplore\" href=\"");
		_out.write("../explorer/show.jsp?");
		_out.write(RdfExplorerUtil.getQueryString(_repository, value, false));
		_out.write("\">");
		_out.write(XmlUtil.escapeCharacterData(RdfExplorerUtil.getDisplayString(value)));
		_out.write("</a>");
	}

	public void error(QueryErrorType errType, String msg)
		throws IOException
	{
		if (!_tableOpen) {
			startTableQueryResult();
		}
		else if (_tupleOpen) {
			endTuple();
		}

		_out.write("<tr><td");
		if (_nrOfColumns > 0) {
			_out.write(" colspan="+_nrOfColumns + ">");
		}
		else {
			_out.write(">");
		}
		_out.write("<font color=\"red\">");

		if (errType == QueryErrorType.MALFORMED_QUERY_ERROR) {
			_out.write("Malformed query error: ");
		}
		else if (errType == QueryErrorType.QUERY_EVALUATION_ERROR) {
			_out.write("Query evaluation error: ");
		}
		else {
			_out.write("Error: ");
		}

		_out.write(XmlUtil.escapeCharacterData(msg));
		_out.write("</font></td></tr>\n");
	}

	protected void endTable() 
		throws IOException
	{
		_out.write("</table>\n");
		_tableOpen = false;
	}
}
