/*  Sesame - Storage and Querying architecture for RDF and RDF Schema
 *  Copyright (C) 2001-2006 Aduna
 *
 *  Contact: 
 *  	Aduna
 *  	Prinses Julianaplein 14 b
 *  	3817 CS Amersfoort
 *  	The Netherlands
 *  	tel. +33 (0)33 465 99 87
 *  	fax. +33 (0)33 465 99 87
 *
 *  	http://aduna-software.com/
 *  	http://www.openrdf.org/
 *  
 *  This library is free software; you can redistribute it and/or
 *  modify it under the terms of the GNU Lesser General Public
 *  License as published by the Free Software Foundation; either
 *  version 2.1 of the License, or (at your option) any later version.
 *
 *  This library is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 *  Lesser General Public License for more details.
 *
 *  You should have received a copy of the GNU Lesser General Public
 *  License along with this library; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

package org.openrdf.sesame.query;

import java.io.IOException;
import java.util.HashSet;
import java.util.Set;

import org.openrdf.model.Resource;
import org.openrdf.model.Statement;
import org.openrdf.model.URI;
import org.openrdf.model.Value;
import org.openrdf.model.impl.StatementImpl;

/**
 * A GraphQueryResultListener that filters any duplicates from the query
 * results and forwards these filtered results to another
 * GraphQueryResultListener.
 **/
public class GraphQueryDuplicatesFilter implements GraphQueryResultListener {

/*------------------------------------------------+
| Variables                                       |
+------------------------------------------------*/

	/** Recorded query result. **/
	protected Set _queryResult;

	/** The listener to forward the filtered query results to. **/
	protected GraphQueryResultListener _listener;

/*------------------------------------------------+
| Constructors                                    |
+------------------------------------------------*/

	/**
	 * Creates a new GraphQueryDuplicatesFilter.
	 * @param l The GraphQueryResultListener that will receive the
	 * filtered query results.
	 **/
	public GraphQueryDuplicatesFilter(GraphQueryResultListener l) {
		_listener = l;
		_queryResult = new HashSet();
	}

/*------------------------------------------------+
| Methods from interface GraphQueryResultListener |
+------------------------------------------------*/

	public void startGraphQueryResult()
		throws IOException
	{
		_listener.startGraphQueryResult();
	}

	public void endGraphQueryResult()
		throws IOException
	{
		_listener.endGraphQueryResult();
		_queryResult.clear();
	}

	public void namespace(String prefix, String name)
		throws IOException
	{
		_listener.namespace(prefix, name);
	}

	public void triple(Resource subj, URI pred, Value obj)
		throws IOException
	{
		Statement statement = new StatementImpl(subj, pred, obj);

		if (!_queryResult.contains(statement)) {
			// new unique result
			_queryResult.add(statement);

			// forward the result
			_listener.triple(subj, pred, obj);
		}
	}

	public void reportError(String msg)
		throws IOException
	{
		_listener.reportError(msg);
	}
}
