/*  Sesame - Storage and Querying architecture for RDF and RDF Schema
 *  Copyright (C) 2001-2006 Aduna
 *
 *  Contact: 
 *  	Aduna
 *  	Prinses Julianaplein 14 b
 *  	3817 CS Amersfoort
 *  	The Netherlands
 *  	tel. +33 (0)33 465 99 87
 *  	fax. +33 (0)33 465 99 87
 *
 *  	http://aduna-software.com/
 *  	http://www.openrdf.org/
 *  
 *  This library is free software; you can redistribute it and/or
 *  modify it under the terms of the GNU Lesser General Public
 *  License as published by the Free Software Foundation; either
 *  version 2.1 of the License, or (at your option) any later version.
 *
 *  This library is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 *  Lesser General Public License for more details.
 *
 *  You should have received a copy of the GNU Lesser General Public
 *  License along with this library; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */
package org.openrdf.sesame.query;

import java.io.DataOutputStream;
import java.io.IOException;
import java.io.OutputStream;
import java.util.HashMap;
import java.util.Map;

import org.openrdf.model.BNode;
import org.openrdf.model.Literal;
import org.openrdf.model.URI;
import org.openrdf.model.Value;

/**
 * Writer for the binary table result format. The format is explained in
 * <tt>BinaryTableResultConstants</tt>.
 *
 * @see BinaryTableResultConstants
 * @author Arjohn Kampman
 * @version $Revision: 1.3.4.2 $
 */
public class BinaryTableResultWriter implements BinaryTableResultConstants, TableQueryResultListener {

/*----------+
| Variables |
+----------*/

	/** The output stream to write the results table to. **/
	private DataOutputStream _out;

	/**
	 * Map containing the namespace IDs (Integer objects) that have been defined
	 * in the document, stored using the concerning namespace (Strings).
	 **/
	private Map _namespaceTable = new HashMap(32);
	private int _nextNamespaceID;

	private Value[] _previousRow;
	private int _columnIdx;

/*-------------+
| Constructors |
+-------------*/

	/**
	 * Creates a new writer for the binary result format that will write the
	 * results to the supplied <tt>OutputStream</tt>.
	 */
	public BinaryTableResultWriter(OutputStream out) {
		_out = new DataOutputStream(out);
	}

/*--------+
| Methods |
+--------*/

	// Implements TableQueryResultListener.startTableQueryResult()
	public void startTableQueryResult()
		throws IOException
	{
		throw new RuntimeException("Column headers must be specified for binary table results.");
	}

	// Implements TableQueryResultListener.startTableQueryResult(String[])
	public void startTableQueryResult(String[] columnHeaders) 
		throws IOException
	{
		_out.write(MAGIC_NUMBER);
		_out.writeInt(FORMAT_VERSION);
		_out.writeInt(columnHeaders.length);

		for (int i = 0; i < columnHeaders.length; i++) {
			_out.writeUTF(columnHeaders[i]);
		}

		_previousRow = new Value[columnHeaders.length];
		_nextNamespaceID = 0;
	}

	// Implements TableQueryResultListener.endTableQueryResult()
	public void endTableQueryResult()
		throws IOException
	{
		_out.writeByte(TABLE_END_RECORD_MARKER);
		_out.flush();
	}

	// Implements TableQueryResultListener.startTuple()
	public void startTuple()
		throws IOException
	{
		_columnIdx = -1;
	}

	// Implements TableQueryResultListener.endTuple()
	public void endTuple()
		throws IOException
	{
	}

	// Implements TableQueryResultListener.tupleValue(Value)
	public void tupleValue(Value value)
		throws IOException
	{
		_columnIdx++;

		if (value == null) {
			_writeNull();
		}
		else if (value.equals(_previousRow[_columnIdx])) {
			_writeRepeat();
		}
		else if (value instanceof URI) {
			_writeQName( (URI)value );
		}
		else if (value instanceof BNode) {
			_writeBNode( (BNode)value );
		}
		else if (value instanceof Literal) {
			_writeLiteral( (Literal)value );
		}
		else {
			throw new IOException("Unknown Value object type: " + value.getClass());
		}

		_previousRow[_columnIdx] = value;
	}

	private void _writeNull()
		throws IOException
	{
		_out.writeByte(NULL_RECORD_MARKER);
	}

	private void _writeRepeat()
		throws IOException
	{
		_out.writeByte(REPEAT_RECORD_MARKER);
	}

	private void _writeQName(URI uri)
		throws IOException
	{
		// Check if the URI has a new namespace
		String namespace = uri.getNamespace();

		Integer nsID = (Integer)_namespaceTable.get(namespace);

		if (nsID == null) {
			// Generate a ID for this new namespace
			nsID = _writeNamespace(namespace);
		}

		_out.writeByte(QNAME_RECORD_MARKER);
		_out.writeInt(nsID.intValue());
		_out.writeUTF(uri.getLocalName());
	}

	private void _writeBNode(BNode bnode)
		throws IOException
	{
		_out.writeByte(BNODE_RECORD_MARKER);
		_out.writeUTF(bnode.getID());
	}

	private void _writeLiteral(Literal literal)
		throws IOException
	{
		String label = literal.getLabel(); 
		String language = literal.getLanguage();
		URI datatype = literal.getDatatype();

		int marker = PLAIN_LITERAL_RECORD_MARKER;

		if (datatype != null) {
			 String namespace = datatype.getNamespace();

			 if (!_namespaceTable.containsKey(namespace)) {
				// Assign an ID to this new namespace
				_writeNamespace(namespace);
			 }

			 marker = DATATYPE_LITERAL_RECORD_MARKER;
		}
		else if (language != null) {
			 marker = LANG_LITERAL_RECORD_MARKER;
		}

		_out.writeByte(marker);
		_out.writeUTF(label);

		if (datatype != null) {
			_writeQName(datatype);
		}
		else if (language != null) {
			_out.writeUTF(language);
		}
	}

	// Implements TableQueryResultListener.tupleValue(Value)
	public void error(QueryErrorType errType, String msg)
		throws IOException
	{
		_out.writeByte(ERROR_RECORD_MARKER);

		if (errType == QueryErrorType.MALFORMED_QUERY_ERROR) {
			_out.writeByte(MALFORMED_QUERY_ERROR);
		}
		else {
			_out.writeByte(QUERY_EVALUATION_ERROR);
		}

		_out.writeUTF(msg);
	}		

	private Integer _writeNamespace(String namespace)
		throws IOException
	{
		_out.writeByte(NAMESPACE_RECORD_MARKER);
		_out.writeInt(_nextNamespaceID);
		_out.writeUTF(namespace);

		Integer result = new Integer(_nextNamespaceID);
		_namespaceTable.put(namespace, result);

		_nextNamespaceID++;

		return result;
	}
}
