/*  Sesame - Storage and Querying architecture for RDF and RDF Schema
 *  Copyright (C) 2001-2006 Aduna
 *
 *  Contact: 
 *  	Aduna
 *  	Prinses Julianaplein 14 b
 *  	3817 CS Amersfoort
 *  	The Netherlands
 *  	tel. +33 (0)33 465 99 87
 *  	fax. +33 (0)33 465 99 87
 *
 *  	http://aduna-software.com/
 *  	http://www.openrdf.org/
 *  
 *  This library is free software; you can redistribute it and/or
 *  modify it under the terms of the GNU Lesser General Public
 *  License as published by the Free Software Foundation; either
 *  version 2.1 of the License, or (at your option) any later version.
 *
 *  This library is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 *  Lesser General Public License for more details.
 *
 *  You should have received a copy of the GNU Lesser General Public
 *  License along with this library; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

package org.openrdf.sesame.query;

import java.io.DataInputStream;
import java.io.IOException;
import java.io.InputStream;
import java.util.Arrays;

import org.openrdf.util.io.IOUtil;

import org.openrdf.model.BNode;
import org.openrdf.model.Literal;
import org.openrdf.model.URI;
import org.openrdf.model.Value;
import org.openrdf.model.ValueFactory;
import org.openrdf.model.impl.ValueFactoryImpl;

/**
 * Reader for the binary table result format. The format is explained in
 * <tt>BinaryTableResultConstants</tt>.
 *
 * @see BinaryTableResultConstants
 * @author Arjohn Kampman
 * @version $Revision: 1.3.4.2 $
 **/
public class BinaryTableResultReader implements BinaryTableResultConstants {

/*--------------+
| Variables     |
+--------------*/

	private ValueFactory _valFactory;
	private TableQueryResultListener _listener;
	private DataInputStream _in;

	private int _columnIdx;
	private int _columnCount;

	private Value[] _previousRow;

	private String[] _namespaceArray = new String[32];

/*--------------+
| Constructors  |
+--------------*/

	/**
	 * Creates a new reader for the binary result format that will use an
	 * instance of <tt>org.openrdf.model.impl.ValueFactoryImpl</tt> to create
	 * Value objects.
	 **/
	public BinaryTableResultReader() {
		this(new ValueFactoryImpl());
	}

	/**
	 * Creates a new reader for the binary result format that will use the
	 * supplied <tt>ValueFactory</tt> to create Value objects.
	 **/
	public BinaryTableResultReader(ValueFactory valueFactory) {
		_valFactory = valueFactory;
	}

/*--------------+
| Methods       |
+--------------*/

	/**
	 * Reads a stream containing a binary table result and reports the parsed
	 * results to the supplied <tt>TableQueryResultListener</tt>.
	 **/
	public synchronized void read(InputStream in, TableQueryResultListener listener)
		throws IOException
	{
		if (in == null) {
			throw new IllegalArgumentException("Input stream can not be 'null'");
		}
		if (listener == null) {
			throw new IllegalArgumentException("listener can not be 'null'");
		}

		_in = new DataInputStream(in);
		_listener = listener;

		byte[] magicNumber = IOUtil.readBytes(_in, MAGIC_NUMBER.length);
		if (!Arrays.equals(magicNumber, MAGIC_NUMBER)) {
			throw new IOException("File does not contain a binary RDF table result");
		}

		int formatVersion = _in.readInt();
		if (formatVersion != FORMAT_VERSION) {
			throw new IOException("Incompatible format version: " + formatVersion);
		}

		_columnCount = _in.readInt();
		if (_columnCount < 1) {
			throw new IOException("Illegal column count specified: " + _columnCount);
		}

		// Read column headers
		String[] columnHeaders = new String[_columnCount];
		for (int i = 0; i < _columnCount; i++) {
			columnHeaders[i] = _in.readUTF();
		}
		_listener.startTableQueryResult(columnHeaders);

		_previousRow = new Value[_columnCount];
		_columnIdx = 0;

		int recordTypeMarker = _in.readByte();

		while (recordTypeMarker != TABLE_END_RECORD_MARKER) {
			if (recordTypeMarker == ERROR_RECORD_MARKER) {
				_processError();
			}
			else if (recordTypeMarker == NAMESPACE_RECORD_MARKER) {
				_processNamespace();
			}
			else {
				Value value;
				switch (recordTypeMarker) {
					case NULL_RECORD_MARKER            : value = null; break;
					case REPEAT_RECORD_MARKER          : value = _previousRow[_columnIdx]; break;
					case QNAME_RECORD_MARKER           : value = _readQName(); break;
					case URI_RECORD_MARKER             : value = _readURI(); break;
					case BNODE_RECORD_MARKER           : value = _readBnode(); break;
					case PLAIN_LITERAL_RECORD_MARKER   :
					case LANG_LITERAL_RECORD_MARKER    :
					case DATATYPE_LITERAL_RECORD_MARKER: value = _readLiteral(recordTypeMarker); break;
					default: throw new IOException("Unkown record type: " + recordTypeMarker);
				}

				_previousRow[_columnIdx] = value;

				if (_columnIdx == 0) {
					_listener.startTuple();
				}

				_listener.tupleValue(value);

				_columnIdx++;

				if (_columnIdx == _columnCount) {
					_listener.endTuple();
					_columnIdx = 0;
				}
			}

			recordTypeMarker = _in.readByte();
		}

		_listener.endTableQueryResult();
	}

	private void _processError()
		throws IOException
	{
		byte errTypeFlag = _in.readByte();

		QueryErrorType errType = null;
		if (errTypeFlag == MALFORMED_QUERY_ERROR) {
			errType = QueryErrorType.MALFORMED_QUERY_ERROR;
		}
		else if (errTypeFlag == QUERY_EVALUATION_ERROR) {
			errType = QueryErrorType.QUERY_EVALUATION_ERROR;
		}
		else {
			throw new IOException("Unkown error type: " + errTypeFlag);
		}

		String msg = _in.readUTF();

		_listener.error(errType, msg);
	}
	
	private void _processNamespace()
		throws IOException
	{
		int namespaceID = _in.readInt();
		String namespace = _in.readUTF();

		if (namespaceID >= _namespaceArray.length) {
			int newSize = Math.max(namespaceID, _namespaceArray.length * 2);
			String[] newArray = new String[newSize];
			System.arraycopy(_namespaceArray, 0, newArray, 0, _namespaceArray.length);
			_namespaceArray = newArray;
		}

		_namespaceArray[namespaceID] = namespace;
	}

	private URI _readQName()
		throws IOException
	{
		int nsID = _in.readInt();
		String localName = _in.readUTF();

		return _valFactory.createURI(_namespaceArray[nsID], localName);
	}

	private URI _readURI()
		throws IOException
	{
		String uri = _in.readUTF();

		return _valFactory.createURI(uri);
	}

	private BNode _readBnode()
		throws IOException
	{
		String bnodeID = _in.readUTF();
		return _valFactory.createBNode(bnodeID);
	}

	private Literal _readLiteral(int recordTypeMarker)
		throws IOException
	{
		String label = _in.readUTF();

		if (recordTypeMarker == DATATYPE_LITERAL_RECORD_MARKER) {
			URI datatype = null;

			int dtTypeMarker = _in.readByte();
			switch (dtTypeMarker) {
				case QNAME_RECORD_MARKER: datatype = _readQName(); break;
				case URI_RECORD_MARKER: datatype = _readURI(); break;
				default: throw new IOException("Illegal record type marker for literal's datatype");
			}

			return _valFactory.createLiteral(label, datatype);
		}
		else if (recordTypeMarker == LANG_LITERAL_RECORD_MARKER) {
			String language = _in.readUTF();
			return _valFactory.createLiteral(label, language);
		}
		else {
			return _valFactory.createLiteral(label);
		}
	}
}
