/*  OMM - Ontology Middleware Module
 *  Copyright (C) 2002 OntoText Lab, Sirma AI OOD
 *
 *  Contact:
 *	Sirma AI OOD, OntoText Lab.
 *	38A, Christo Botev Blvd.
 *  1000 Sofia, Bulgaria
 *	tel. +359(2)981 00 18
 *	fax. +359(2)981 90 58
 *	info@ontotext.com
 *
 * 	http://www.ontotext.com/
 *
 *  This library is free software; you can redistribute it and/or
 *  modify it under the terms of the GNU Lesser General Public
 *  License as published by the Free Software Foundation; either
 *  version 2.1 of the License, or (at your option) any later version.
 *
 *  This library is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 *  Lesser General Public License for more details.
 *
 *  You should have received a copy of the GNU Lesser General Public
 *  License along with this library; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */
package org.openrdf.sesame.omm;

/**
 * Title:        OTK: SAIL interface definitions
 * Description:  This code is a part of OnTo-Knowledge project.
 * Company:      Sirma AI - OntoText Lab.
 * @author Damyan Ognyanoff
 * @version 1.0
 */

import java.util.Iterator;
import java.util.Map;

import org.openrdf.sesame.sail.Sail;
/**
 * The interface of provides the basic oprationsto manage the distinct
 * versions and states of the repository. Locking/Unlocking of the statements
 * in the repository can be applyed.
 */
public interface VersionManagement extends Sail {
  /**
   * Create a labeled version for a state of the repository assigning the
   * necessary mata-information about that operation.
   *
   * @param stateUID the update counter of a particular repository state
   * @param label the label to assign
   * NOTE: This method can throw a security exception if the request is made on
   * behalf of the user with insufficent rights to create versions
   */
  void labelState(long stateUID, String label);

  /**
   * Create a labeled version of the curent repository state.
   * NOTE: This method can throw a security exception if the request is made on
   * behalf of the user with insufficent rights to create versions
   */
  void labelCurrentState(String label);

  /**
   * Restore the repository to previous state removing all statements added
   * after the value of the update counter and revive all remover ones.
   * @param stateUID the update counter of a particular repository state
   * NOTE: This method can throw a security exception if the request is made on
   * behalf of the user with insufficent rights to revert the repository
   */
  void revertToState(long stateUID);

  /**
   * Sets the repository to given statete for further read operations.
   * @param stateUID the update counter of a particular repository state
   */
  void workWithState(long stateUID);

  /**
   * branch the repository at given state for further operations.
   * @param stateUID the update counter of a particular repository state
	 * @return the id of the repository
   */
  String branchState(long stateUID);

  /**
   * Retrive list of all labeled states of the repository.
   * @return a list of Versin interfaces for each labeled state of the repository
   */
  Iterator getVersions();

  /**
   * Perform locking of statements in the repository
   * @param statementsList list of statemensts to lock
   */
  void lockStatements(Iterator statementsList);
  /**
   * Perform unlocking of statements in the repository
   * @param statementsList list of statemensts to unlock
   */
  void unlockStatements(Iterator statementsList);
  /**
   * Retrive list of all Updates of the repository.
   * @return a list of Updates of the repository
   */
  Iterator getUpdateIds();

  /**
   * Stop the increment of the update counter. Usefull for a kind of batch updates
   * or adding a distinct daml coinstructs at once.
   */
  public void pauseCounterIncrement();

  /**
   * Coninue with the normal increment of the update counter on each modification
   * made in the repository.
   */
  public void continueCounterIncrement();

  /**
   * Check if the update couter ss paused
   * @return true if the updateCouter is paused, flase otherwise
   */
  public boolean isPausedCounterIncrement();

	/**
	 * Retrieves an iterator over the version ids.
	 * @return an iterator over the version ids
	 */
	Iterator getVersionIds();

	/* MetaInfo Retrieval Methods */


	/**
	 * Retrieves the meta info associated with a statement.
	 * @param subj the subject of the statement
	 * @param pred the predicate of the statement
	 * @param obj the object of the statement
	 * @return a map of meta info keys vs meta info values
	 */
	Map getMetaInfo(String subj, String pred, String obj);

	/**
	 * Retrieves the meta info associated with an update.
	 * @param updateId the id of the update
	 * @return a map of meta info keys vs meta info values
	 */
	Map getUpdateMetaInfo(String updateId);

	/**
	 * Retrieves the meta info associated with a version .
	 * @param versionId the id of the update
	 * @return a map of meta info keys vs meta info values
	 */
	Map getVersionMetaInfo(String versionId);

} //interface VersionManagement
//
