/*  OMM - Ontology Middleware Module
 *  Copyright (C) 2002 OntoText Lab, Sirma AI OOD
 *
 *  Contact:
 *	Sirma AI OOD, OntoText Lab.
 *	38A, Christo Botev Blvd.
 *  1000 Sofia, Bulgaria
 *	tel. +359(2)981 00 18
 *	fax. +359(2)981 90 58
 *	info@ontotext.com
 *
 * 	http://www.ontotext.com/
 *
 *  This library is free software; you can redistribute it and/or
 *  modify it under the terms of the GNU Lesser General Public
 *  License as published by the Free Software Foundation; either
 *  version 2.1 of the License, or (at your option) any later version.
 *
 *  This library is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 *  Lesser General Public License for more details.
 *
 *  You should have received a copy of the GNU Lesser General Public
 *  License along with this library; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */
package org.openrdf.sesame.omm;

 /**
  * <p>Title: Knowledge Control System</p>
  * <p> </p>
  * <p> </p>
  * <p>Company: OntoText Lab. Sirma AI. </p>
  * @author Damyan Ognyanoff, borislav popov
  * @version 1.0
  */
import java.util.Set;

import org.openrdf.model.Resource;
import org.openrdf.model.URI;
import org.openrdf.model.Statement;
import org.openrdf.model.Value;

import org.openrdf.sesame.sailimpl.omm.security.NullParameterException;
import org.openrdf.sesame.sailimpl.omm.security.Role;
import org.openrdf.sesame.sailimpl.omm.security.Restriction;
import org.openrdf.sesame.sailimpl.omm.security.Right;
import org.openrdf.sesame.sailimpl.omm.security.SecurityException;
import org.openrdf.sesame.sailimpl.omm.security.User;


/**
 * Security Services inteface.
 * That inteface holds all external methods to manage users and groups for the
 * repository. This includes user authenication, basic users management, and
 * basic group managemenet methods.
 * Manipulation of a patricular user or group is made via more specific interfaces
 * that can be retrieved through this one.
 * This interface also provides import and export of the security setup from/to RDF.
 */
public interface SecurityServices  {

  /**
   * Adds an new user.
   * @param id user's id
   * @param login user's login
   * @param password user's password
   * @param name user's name
   */
  void addUser(int id, String login, String password,String name);

  /**
   * Removes a user given its login.
   * @param login the login of the user to be removed.
   */
  void removeUser(String login);

  /**
   * Retrieves a set of the users in the repository. Each element of the list is
   * a String containing the user's login.
   * @return set of the users logins.
   */
  Set getUsers();

  /**
   * Retrieves a interface to an object that supports the User interface.
   * @param login the user's login
   * @return the user object represented by the param login, or null if no such user login.
   */
  User getUser(String login);

  /**
   * Gets a user, given it's id
   * @param id the user's id
   * @return the user object represented by the param id, or null if no such user id.
   */
  User getUser(int id);


  /**
   * Creates a new Role in the repository.
   * @param id the id of the role
   * @param name the name of the role to be created
   * @param desctiption the desctiption of the role to be created
   * @param parentRoles Set of parent roles to inherit permissions from.
   * @return the created Role
   */
  Role createRole(int id, String name, String desctiption, Set parentRoles);

  /**
   * Remove user role from the repository.
   * @param name the name of the role to be removed
   */
  void removeRole(String name);

  /**
   * Removes user role from the repository.
   * @param id the id of the role to be removed
   */
  void removeRole(int id);

  /**
   * Retrive a set of the roles' ids.
   * @return set of the roles' ids
   */
  Set getRoles();

  /**
   * Gets the role given its name.
   * @param name the name of the role.
   * @return the role that corresponds to the given name, or null if there is
   * no such known name.
   */
  Role getRole(String name);

  /**
   * Gets the role given its id.
   * @param id the id of the role.
   * @return the role that corresponds to the given id, or null if there is
   * no such known id.
   */
  Role getRole(int id);


  /**
   * Creates a new restriction.
   * @param id
   * @param type type of the restriction
   * @param name
   * @param description
   * @return the creaed restriction
   * @throws NullParameterException if a parameter is null.
   */
  Restriction createRestriction(int id, int type, String name, String description)
  throws NullParameterException;

  /**
   * Gets restriction by id.
   * @param id
   * @return the restriction corresponding to the given id.
   */
  Restriction getRestriction(int id);

  /**
   * Checks the accessibility of a resource according the security policy and the
   * existence of a Read right over the resource.
   * @param res the resource
   * @return true if the resource is accessible, otherwise - false.
   */
  boolean isResourceAccessible(Resource res);

  /**
   * Checks the accessibility of a statement according the security policy and a given right.
   * @param st The statement.
   * @param right the right which is being checked for this statement
   * @return true if the statement is accessible with the given right, otherwise - false.
   */
  boolean isStatementAccessible(Statement st, Right right);


  /**
   * Checks the accessibility of a statement according the security policy and a given right.
   * @param subj the subject of the statement
   * @param pred the predicate of the statement
   * @param obj the object of the statement
   * @param right the right which is being checked for this statement
   * @return true if the statement is accessible with the given right, otherwise - false.
   */
  boolean isStatementAccessible(Resource subj, URI pred, Value obj, Right right);

  /**
   * Checks the accessibility of a value according to the security policy.
   * @param val the value to be checked
   * @return true if the value is accessible through a read right, otherwise - false
   */
  boolean isValueAccessible(Value val);

  /**
   * Checks the accessibility of a repository for a specified right.
   * @param right the right to be checked
   * @return true if the repository is accessible with the given right, otherwise - false.
   */
  boolean isRepositoryAccessible(Right right);


  /**
   * Checks the accessibility of a tracking for a specified right.
   * @param right the right to be checked
   * @return true if the schema is accessible with the given right, otherwise - false.
   */
  boolean isSchemaAccessible(Right right);

  /**
   * Checks if the user has access to the versioning and tracking sail.
   * @return true if the user has assiogned rule with a repository
   * restriction and history right set to true, otherwise - false.
   */
  boolean isVersionTrackingAccessible();

  /**
   * Gets the id associated with a resource.
   * @param res a resource
   * @return the id of the resource, if found
   * @throws SecurityException if something goes wrong with the execution of the method
   */
  int getResourceId(Resource res) throws org.openrdf.sesame.sailimpl.omm.security.SecurityException;

  /**
   * Gets a resource given its id
   * @param id the id of the resource
   * @return the resource with the specified id
   */
  Resource getResource(int id);

  // >>> RDF import and export of the security setup

/* Java depndent methods
  void exportSecurityPolicy2RDF(Writer out);

  void importSecurityPolicy(Reader in);
*/
	/**
	 * Exports the Security Policy to a Sail.
	 * To be used with an In-Memory sail for example
	 * (org.openrdf.sesame.sail.memory.list.RdfRepository).
	 * @param repos the RdfRepository to export to
	 */
	public void exportPolicy(org.openrdf.sesame.sail.RdfRepository repos);

	/**
	 * Imports the security policy given an RDF Graph.
	 * @param graph the Graph to read the security policy from
	 */
	public void importPolicy(org.openrdf.model.Graph graph);

} //Security Services interface


