/*  Sesame - Storage and Querying architecture for RDF and RDF Schema
 *  Copyright (C) 2001-2006 Aduna
 *
 *  Contact:
 *  	Aduna
 *  	Prinses Julianaplein 14 b
 *  	3817 CS Amersfoort
 *  	The Netherlands
 *  	tel. +33 (0)33 465 99 87
 *  	fax. +33 (0)33 465 99 87
 *
 *  	http://aduna-software.com/
 *  	http://www.openrdf.org/
 *
 *  This library is free software; you can redistribute it and/or
 *  modify it under the terms of the GNU Lesser General Public
 *  License as published by the Free Software Foundation; either
 *  version 2.1 of the License, or (at your option) any later version.
 *
 *  This library is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 *  Lesser General Public License for more details.
 *
 *  You should have received a copy of the GNU Lesser General Public
 *  License along with this library; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

package org.openrdf.sesame.export;

import java.io.IOException;
import java.util.HashSet;
import java.util.Iterator;
import java.util.Set;

import org.openrdf.model.Resource;
import org.openrdf.model.Statement;

import org.openrdf.rio.RdfDocumentWriter;

import org.openrdf.sesame.sail.NamespaceIterator;
import org.openrdf.sesame.sail.RdfSchemaSource;
import org.openrdf.sesame.sail.RdfSource;
import org.openrdf.sesame.sail.StatementIterator;

public class RdfExport {

	/**
	 * Creates a new RdfExport that can extract data from a Sail and export
	 * it as RDF.
	 **/
	public RdfExport() {
	}

/*----------------------------------------+
| Methods for RdfSource                   |
+----------------------------------------*/

	/**
	 * Reads statements from the supplied RdfSource and reports these to the
	 * supplied RdfDocumentWriter.
	 * @param rs The RdfSource from which to extract the statements.
	 * @param rdfDocWriter The RdfDocumentWriter that will be used to write
	 * the RDF document.
	 * @param niceOutput If true, extra effort will be invested into making
	 * the exported data more readable to humans (e.g. by sorting statements
	 * by their subject).
	 **/
	public void exportRdf(
		RdfSource rs, RdfDocumentWriter rdfDocWriter, boolean niceOutput)
		throws IOException
	{
		_setNamespaces(rs, rdfDocWriter);

		rdfDocWriter.startDocument();

		_writeStatements(rs, rdfDocWriter, niceOutput);

		rdfDocWriter.endDocument();
	}

	protected void _writeStatements(
		RdfSource rs, RdfDocumentWriter rdfDocWriter, boolean niceOutput)
		throws IOException
	{
		if (niceOutput) {
			// We want to group statements according to their subject. So first
			// we collect all subject (could be a big set...), and then export
			// all of this subject's statements.

			Set subjects = new HashSet();

			StatementIterator statIter = rs.getStatements(null, null, null);
			while (statIter.hasNext()) {
				Statement st = statIter.next();
				subjects.add( st.getSubject() );
			}
			statIter.close();

			// Write all statements, grouped by subject.
			Iterator subjIter = subjects.iterator();
			while (subjIter.hasNext()) {
				Resource subject = (Resource)subjIter.next();
				_writeSubjStatements(rs, rdfDocWriter, subject);
			}
		}
		else {
			// Just export the statements in the order that they are supplied
			// by the RdfSource.
			StatementIterator statIter = rs.getStatements(null, null, null);

			while (statIter.hasNext()) {
				Statement st = statIter.next();

				rdfDocWriter.writeStatement(
						st.getSubject(), st.getPredicate(), st.getObject());
			}

			statIter.close();
		}
	}

	/**
	 * Writes all statements having a specific subject.
	 **/
	private void _writeSubjStatements(
		RdfSource rs, RdfDocumentWriter rdfDocWriter, Resource subj)
		throws IOException
	{
		// Write all statements with 'subj' as subject
		StatementIterator statIter = rs.getStatements(subj, null, null);

		while (statIter.hasNext()) {
			Statement st = statIter.next();

			rdfDocWriter.writeStatement(subj, st.getPredicate(), st.getObject());
		}

		statIter.close();
	}

/*----------------------------------------+
| Methods for RdfSchemaSource             |
+----------------------------------------*/

	/**
	 * Reads statements from the supplied RdfSchemaSource and reports these to
	 * the supplied RdfDocumentWriter.
	 * @param rss The RdfSchemaSource from which to extract the statements.
	 * @param rdfDocWriter The RdfDocumentWriter that will be used to write
	 * the RDF document.
	 * @param exportOnto If true, statements defining classes and properties
	 * will be exported.
	 * @param exportData If true, statements not defining classes and properties
	 * will be exported.
	 * @param explicitOnly If true, only explicit statements will be exported.
	 * @param niceOutput If true, extra effort will be invested into making
	 * the exported data more readable to humans (e.g. by sorting statements
	 * by their subject).
	 **/
	public void exportRdf(
		RdfSchemaSource rss, RdfDocumentWriter rdfDocWriter,
		boolean exportOnto, boolean exportData,
		boolean explicitOnly, boolean niceOutput)
		throws IOException
	{
		_setNamespaces(rss, rdfDocWriter);

		rdfDocWriter.startDocument();

		if (exportOnto && exportData) {
			_writeStatements(rss, rdfDocWriter, explicitOnly, niceOutput);
		}
		else if (exportOnto) {
			_writeClassDefs(rss, rdfDocWriter, explicitOnly, niceOutput);
			_writePropertyDefs(rss, rdfDocWriter, explicitOnly, niceOutput);
		}
		else if (exportData) {
			_writeDataStatements(rss, rdfDocWriter, explicitOnly, niceOutput);
		}

		rdfDocWriter.endDocument();
	}

	protected void _writeStatements(
		RdfSchemaSource rss, RdfDocumentWriter rdfDocWriter,
		boolean explicitOnly, boolean niceOutput)
		throws IOException
	{
		if (niceOutput) {
			rdfDocWriter.writeComment("All statement");

			// We want to group statements according to their subject. So first
			// we collect all subject (could be a big set...), and then export
			// all of this subject's statements.

			Set subjects = new HashSet();

			StatementIterator statIter;
			if (explicitOnly) {
				statIter = rss.getExplicitStatements(null, null, null);
			}
			else {
				statIter = rss.getStatements(null, null, null);
			}

			while (statIter.hasNext()) {
				Statement st = statIter.next();
				subjects.add( st.getSubject() );
			}

			statIter.close();

			// Write all statements, grouped by subject.
			Iterator subjIter = subjects.iterator();
			while (subjIter.hasNext()) {
				Resource subject = (Resource)subjIter.next();
				_writeSubjStatements(rss, rdfDocWriter, explicitOnly, subject);
			}
		}
		else {
			// Just export the statements in the order that they are supplied
			// by the RdfSchemaSource.
			StatementIterator statIter;
			if (explicitOnly) {
				statIter = rss.getExplicitStatements(null, null, null);
			}
			else {
				statIter = rss.getStatements(null, null, null);
			}

			while (statIter.hasNext()) {
				Statement st = statIter.next();

				rdfDocWriter.writeStatement(
						st.getSubject(), st.getPredicate(), st.getObject());
			}

			statIter.close();
		}
	}

	protected void _writeClassDefs(
		RdfSchemaSource rss, RdfDocumentWriter rdfDocWriter,
		boolean explicitOnly, boolean niceOutput)
		throws IOException
	{
		if (niceOutput) {
			rdfDocWriter.writeComment("Class definitions");
		}

		// Get the set of all classes

		StatementIterator classIter = rss.getClasses();

		while (classIter.hasNext()) {
			Resource classRes = (Resource)classIter.next().getSubject();

			// Export all statements about this class.
			_writeSubjStatements(rss, rdfDocWriter, explicitOnly, classRes);
		}

		classIter.close();
	}

	protected void _writePropertyDefs(
		RdfSchemaSource rss, RdfDocumentWriter rdfDocWriter,
		boolean explicitOnly, boolean niceOutput)
		throws IOException
	{
		if (niceOutput) {
			rdfDocWriter.writeComment("Property definitions");
		}

		// Get the set of all properties.

		StatementIterator propIter = rss.getProperties();

		while (propIter.hasNext()) {
			Resource property = (Resource)propIter.next().getSubject();

			// Export all statements about this property.
			_writeSubjStatements(rss, rdfDocWriter, explicitOnly, property);
		}

		propIter.close();
	}

	protected void _writeDataStatements(
		RdfSchemaSource rss, RdfDocumentWriter rdfDocWriter,
		boolean explicitOnly, boolean niceOutput)
		throws IOException
	{
		// Data statements are all statements for which the subject is
		// not a class or property

		if (niceOutput) {
			rdfDocWriter.writeComment("Data statements");

			// Collect all subjects.
			Set subjects = new HashSet();

			StatementIterator statIter;
			if (explicitOnly) {
				statIter = rss.getExplicitStatements(null, null, null);
			}
			else {
				statIter = rss.getStatements(null, null, null);
			}

			while (statIter.hasNext()) {
				Statement st = statIter.next();
				subjects.add( st.getSubject() );
			}
			statIter.close();

			// Remove all subjects that are classes
			StatementIterator classIter = rss.getClasses();
			while (classIter.hasNext()) {
				subjects.remove( classIter.next().getSubject() );
			}
			classIter.close();

			// Remove all subjects that are properties
			StatementIterator propIter = rss.getProperties();
			while (propIter.hasNext()) {
				subjects.remove( propIter.next().getSubject() );
			}
			propIter.close();

			// Write all data statements, grouped by subject.
			Iterator subjIter = subjects.iterator();
			while (subjIter.hasNext()) {
				Resource subject = (Resource)subjIter.next();
				_writeSubjStatements(rss, rdfDocWriter, explicitOnly, subject);
			}
		}
		else {
			// Write all statements that are not about classes or properties
			// in arbitrary order.


			// Collect all classes and properties.
			Set classesAndProperties = new HashSet();

			StatementIterator classIter = rss.getClasses();
			while (classIter.hasNext()) {
				classesAndProperties.add( classIter.next().getSubject() );
			}
			classIter.close();

			StatementIterator propIter = rss.getProperties();
			while (propIter.hasNext()) {
				classesAndProperties.add( propIter.next().getSubject() );
			}
			propIter.close();

			// Extract all statements and export the ones whose subject is
			// not a class or property.
			StatementIterator statIter;
			if (explicitOnly) {
				statIter = rss.getExplicitStatements(null, null, null);
			}
			else {
				statIter = rss.getStatements(null, null, null);
			}

			while (statIter.hasNext()) {
				Statement st = statIter.next();

				Resource subject = st.getSubject();

				if (!classesAndProperties.contains(subject)) {
					rdfDocWriter.writeStatement(
							subject, st.getPredicate(), st.getObject());
				}
			}

			statIter.close();
		}
	}

	/**
	 * Writes all statements having a specific subject.
	 **/
	private void _writeSubjStatements(RdfSchemaSource rss, RdfDocumentWriter rdfDocWriter,
		boolean explicitOnly, Resource subj)
		throws IOException
	{
		// Write all statements with 'subj' as subject
		StatementIterator statIter;
		if (explicitOnly) {
			statIter = rss.getExplicitStatements(subj, null, null);
		}
		else {
			statIter = rss.getStatements(subj, null, null);
		}

		while (statIter.hasNext()) {
			Statement st = statIter.next();

			rdfDocWriter.writeStatement(subj, st.getPredicate(), st.getObject());
		}

		statIter.close();
	}

/*-------------------------------------------------------+
| Generic methods for both RdfSource and RdfSchemaSource |
+-------------------------------------------------------*/

	private void _setNamespaces(RdfSource rs, RdfDocumentWriter rdfDocWriter)
		throws IOException
	{
		NamespaceIterator nsIter = rs.getNamespaces();

		while (nsIter.hasNext()) {
			nsIter.next();

			String prefix = nsIter.getPrefix();
			String name = nsIter.getName();

			if (prefix != null) {
				rdfDocWriter.setNamespace(prefix, name);
			}
		}

		nsIter.close();
	}
}
