package org.openrdf.sesame.config.ui.util;

import java.awt.Component;
import java.awt.Container;
import java.awt.GridBagConstraints;
import java.awt.GridBagLayout;
import java.awt.Insets;

/**
 * Facilitates the use of GridBagLayout.
 * @see java.awt.GridBagLayout
 * @see java.awt.GridBagConstraints
 */
public class GridBagUtil {

	/**
	 * Constrain a component to a location and size in the specified
	 * container.
	 * @param cont the container in which the component is to be
	 * constrained.
	 * @param comp the component to constrain.
	 * @param grid_x the x-coordinate of the component.
	 * @param grid_y the y-coordinate of the component.
	 * @param grid_width the width of the component.
	 * @param grid_height the height of the component.
	 */
	public static void constrain(Container cont, Component comp,
			int grid_x, int grid_y, int grid_width, int grid_height)
	{
		constrain(cont, comp,
			grid_x, grid_y, grid_width, grid_height,
			GridBagConstraints.NONE, GridBagConstraints.CENTER, 0.0, 0.0,
			0, 0, 0, 0);
	}

	/**
	 * Constrain a component to a location and size in the specified
	 * container, also setting a certain amount of space around it.
	 * @param cont the container in which the component is to be
	 * constrained.
	 * @param comp the component to constrain.
	 * @param grid_x the x-coordinate of the component.
	 * @param grid_y the y-coordinate of the component.
	 * @param grid_width the width of the component.
	 * @param grid_height the height of the component.
	 * @param top the amount of space above the component (in pixels).
	 * @param left the amount of space to the left of the component
	 * (in pixels).
	 * @param bottom the amount of space below the component
	 * (in pixels).
	 * @param right the amount of space to the right of the component
	 * (in pixels).
	 */
	public static void constrain(Container cont, Component comp,
		int grid_x, int grid_y, int grid_width, int grid_height,
		int top, int left, int bottom, int right)
	{
		constrain(cont, comp,
			grid_x, grid_y, grid_width, grid_height,
			GridBagConstraints.NONE, GridBagConstraints.CENTER, 0.0, 0.0,
			top, left, bottom, right);
	}

	/**
	 * Constrain a component in a container.
	 * @param cont the container in which the component is to be
	 * constrained.
	 * @param comp the component to constrain.
	 * @param grid_x the x-coordinate of the component.
	 * @param grid_y the y-coordinate of the component.
	 * @param grid_width the width of the component.
	 * @param grid_height the height of the component.
	 * @param fill   how to fill out the component.
	 * @param anchor  where to anchor the component.
	 * @param weight_x  the x-weight of the component.
	 * @param weight_y  the y-weight of the component.
	 * @param top   the amount of space above the component (in pixels).
	 * @param left   the amount of space to the left of the component
	 * (in pixels).
	 * @param bottom  the amount of space below the component
	 * (in pixels).
	 * @param right  the amount of space to the right of the component
	 * (in pixels).
	 */
	public static void constrain(Container cont, Component comp,
		int grid_x, int grid_y, int grid_width, int grid_height,
		int fill, int anchor, double weight_x, double weight_y,
		int top, int left, int bottom, int right)
	{
		GridBagConstraints c = new GridBagConstraints();
		c.gridx = grid_x;
		c.gridy = grid_y;
		c.gridwidth = grid_width;
		c.gridheight = grid_height;
		c.fill = fill;
		c.anchor = anchor;
		c.weightx = weight_x;
		c.weighty = weight_y;

		if (top+bottom+left+right > 0) {
			c.insets = new Insets(top, left, bottom, right);
		}

		((GridBagLayout)cont.getLayout()).setConstraints(comp, c);
		cont.add(comp);
	}
}
