/*  Sesame - Storage and Querying architecture for RDF and RDF Schema
 *  Copyright (C) 2001-2006 Aduna
 *
 *  Contact:
 *  	Aduna
 *  	Prinses Julianaplein 14 b
 *  	3817 CS Amersfoort
 *  	The Netherlands
 *  	tel. +33 (0)33 465 99 87
 *  	fax. +33 (0)33 465 99 87
 *
 *  	http://aduna-software.com/
 *  	http://www.openrdf.org/
 *
 *  This library is free software; you can redistribute it and/or
 *  modify it under the terms of the GNU Lesser General Public
 *  License as published by the Free Software Foundation; either
 *  version 2.1 of the License, or (at your option) any later version.
 *
 *  This library is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 *  Lesser General Public License for more details.
 *
 *  You should have received a copy of the GNU Lesser General Public
 *  License along with this library; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

package org.openrdf.sesame.config.ui;

import java.util.ArrayList;
import java.util.List;

import javax.swing.table.AbstractTableModel;

import org.openrdf.sesame.config.SystemConfig;
import org.openrdf.sesame.config.SystemConfigListener;

/**
 * WRITEME
 *
 * @author Peter van 't Hof
 * @author Arjohn Kampman
 * @version $Revision: 1.4.4.2 $
 **/
public abstract class XTableModel extends AbstractTableModel
	implements SystemConfigListener
{

/*----------+
| Variables |
+----------*/

	/** SystemConfig */
	protected SystemConfig _config;

	/**
	 * An array of ColumnData objects containing meta-info
	 * about the table's columns.
	 **/
	protected ColumnData[] _columns;

	/**
	 * A List of RowData objects.
	 **/
	protected List _rows;

/*-------------+
| Constructors |
+-------------*/

	/**
	 * Creates a new XTableModel having the specified columns.
	 **/
	public XTableModel(SystemConfig config, ColumnData[] columns) {
		_config = config;
		_columns = columns;
		_rows = new ArrayList();

		_config.addListener(this);
	}

/*--------+
| Methods |
+--------*/

	/**
	 * By default every cell in an XTableModel is editable.
	 **/
	public boolean isCellEditable(int row, int column) {
		return true;
	}

	public int getColumnCount() {
		return _columns.length;
	}

	// Overrides AbstractTableModel.getColumnName(int)
	public String getColumnName(int column) {
		return _columns[column].title;
	}

	public int getColumnWidth(int column) {
		return _columns[column].width;
	}

	public int getColumnAlignment(int column) {
		return _columns[column].alignment;
	}

	/**
	 * Gets the index of the identifying column. Every row in a table is
	 * identified by the value of a specific column, e.g. the user table
	 * is identified by its login column, the repository table is
	 * identified by its repositoryID column, etc.
	 *
	 * @return The index of the identifying column (0-based).
	 **/
	public abstract int getIdentifyingColumn();

	public int getRowCount() {
		return _rows.size();
	}

	/**
	 * Gets the index of the row having the specified value as its identifier.
	 *
	 * @param rowID A value for the identifying column in the table.
	 * @return The row index (0-based), or -1 if the ID was not found.
	 **/
	public int getRowIndex(String rowID) {
		for (int i = 0; i < _rows.size(); i++) {
			RowData row = (RowData)_rows.get(i);

			if (rowID.equals(row.getIdentifier())) {
				return i;
			}
		}

		return -1;
	}

	/**
	 * Adds a new row at the supplied row index
	 *
	 * @param rowIdx The index of the new row in the table.
	 **/
	public void addNewRow(int rowIdx) {
		_rows.add(rowIdx, _createRow());

		fireTableRowsInserted(rowIdx, rowIdx);
	}

	/**
	 * Creates a new RowData object.
	 **/
	protected abstract RowData _createRow();

	/** Removes the new row */
	public void removeNewRow() {
		int size = _rows.size();

		for (int i = 0; i < size; i ++ ) {
			RowData row = (RowData)_rows.get(i);

			if (row.isNew()) {
				_rows.remove(i);

				fireTableRowsDeleted(i, i);

				return;
			}
		}
	}

	/**
	 * Checks if the row at the supplied row index is new.
	 *
	 * @param rowIdx An row index.
	 * @return <tt>true</tt> if the row was new, <tt>false</tt> otherwise.
	 */
	public boolean rowIsNew(int rowIdx) {
		RowData row = (RowData)_rows.get(rowIdx);
		return row.isNew();
	}

	/**
	 * Checks if the value at the supplied row and column index is new.
	 *
	 * @param row Row index
	 * @param column Column index
	 * @return Boolean indicating if the value is new
	 */
	public boolean valueIsNew(int row, int column) {
		return getValueAt(row, column) == null;
	}

/*-----------------------------+
| Inner class RowData          |
+-----------------------------*/

	/** Row of XTableModel */
	protected abstract class RowData implements Comparable {

		/**
		 * Gets the value of the identifying column. Every row is identified by
		 * a value, e.g a user is identified by its user login, a repository by
		 * its repository id, etc.
		 *
		 * @return The value of the identifying column.
		 */
		public abstract String getIdentifier();

		/**
		 * Checks if this row is new
		 *
		 * @return <tt>true</tt> if this row is new, <tt>false</tt> otherwise.
		 */
		public abstract boolean isNew();

		/**
		 * Rows are compared using their identifier by default.
		 *
		 * @see Comparable.compareTo
		 */
		public int compareTo(Object other) {
			RowData row = (RowData)other;
			return getIdentifier().compareTo(row.getIdentifier());
		}
	}
}
