/*  Sesame - Storage and Querying architecture for RDF and RDF Schema
 *  Copyright (C) 2001-2006 Aduna
 *
 *  Contact:
 *  	Aduna
 *  	Prinses Julianaplein 14 b
 *  	3817 CS Amersfoort
 *  	The Netherlands
 *  	tel. +33 (0)33 465 99 87
 *  	fax. +33 (0)33 465 99 87
 *
 *  	http://aduna-software.com/
 *  	http://www.openrdf.org/
 *
 *  This library is free software; you can redistribute it and/or
 *  modify it under the terms of the GNU Lesser General Public
 *  License as published by the Free Software Foundation; either
 *  version 2.1 of the License, or (at your option) any later version.
 *
 *  This library is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 *  Lesser General Public License for more details.
 *
 *  You should have received a copy of the GNU Lesser General Public
 *  License along with this library; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

package org.openrdf.sesame.config.ui;

import java.awt.Component;
import java.awt.Window;

import javax.swing.JTable;
import javax.swing.ListSelectionModel;
import javax.swing.event.ChangeEvent;
import javax.swing.event.ListSelectionEvent;
import javax.swing.event.TableModelEvent;
import javax.swing.table.DefaultTableCellRenderer;
import javax.swing.table.JTableHeader;
import javax.swing.table.TableCellEditor;
import javax.swing.table.TableCellRenderer;
import javax.swing.table.TableColumn;

import org.openrdf.sesame.config.SystemConfig;
import org.openrdf.sesame.config.ui.util.Util;

/**
 * WRITEME
 *
 * @author Peter van 't Hof
 * @author Arjohn Kampman
 * @version $Revision: 1.4.4.2 $
 */
public abstract class XTable extends JTable {

/*----------+
| Variables |
+----------*/

	/** SystemConfig */
	protected SystemConfig _config;

	/** The model for this XTable. **/
	protected XTableModel _model;

	/** Edited value */
	protected String _value;

/*-------------+
| Constructors |
+-------------*/

	public XTable(SystemConfig config) {
		_config = config;
	}

/*--------+
| Methods |
+--------*/

	public void setXTableModel(XTableModel model) {
		_model = model;

		// Create own colums
		setAutoCreateColumnsFromModel(false);

		setModel(model);

		for (int i = 0; i < model.getColumnCount(); i++) {
			TableCellRenderer renderer = _createCellRenderer(i);
			TableCellEditor editor = _createCellEditor(i);

			TableColumn column = new TableColumn(i, model.getColumnWidth(i), renderer, editor);

			// FIXME
			//
			// Set the alignment of the header TableCellRenderer for this
			// TableColumn. Default, the header renderer is for each column null.
			// Therefore, install the default header renderer of a JTableHeader
			// as the header renderer of this column. The only way to create a
			// new default header renderer is to create a new header and get its
			// default renderer.
			JTableHeader header = new JTableHeader();
 			DefaultTableCellRenderer headerRenderer =
					(DefaultTableCellRenderer)header.getDefaultRenderer();
			
			headerRenderer.setHorizontalAlignment(model.getColumnAlignment(i));
			
 			column.setHeaderRenderer(headerRenderer);

			addColumn(column);
		}

		getTableHeader().setReorderingAllowed(false);

		// setShowGrid(false);

		setSelectionMode(ListSelectionModel.SINGLE_SELECTION);

		// getColumnModel().setColumnMargin(0);
		// seAutoResizeMode(JTable.AUTO_RESIZE_OFF);
	}

	/**
	 * Creates a TableCellRenderer for column <tt>columnNo</tt>.
	 **/
	protected abstract TableCellRenderer _createCellRenderer(int columnNo);

	/**
	 * Creates a TableCellEditor for column <tt>columnNo</tt>.
	 **/
	protected abstract TableCellEditor _createCellEditor(int columnNo);

	/**
	 * Sets the the edited value
	 *
	 * @see javax.swing.JTable#editingStopped
	 */
	public void editingStopped(ChangeEvent e) {
		if (getEditingColumn() == _model.getIdentifyingColumn()) {
			// Id column is currently being edited. As the table is not updated
			// yet, there is no value available from the model. Therefore get
			// the value from the editor.
			_value = (String)getCellEditor().getCellEditorValue();
		}
		else {
			_value = getIdentifierForSelectedRow();
		}

		super.editingStopped(e);
	}

	/**
	 * Sets the value of the XCellEditor
	 *
	 * @see javax.swing.JTable#prepareEditor
	 */
	public Component prepareEditor(TableCellEditor editor, int row, int column) {
		// Reset value
		_value = null;

		Object value = getValueAt(row, column);
		XCellEditor xCellEditor = (XCellEditor)editor;

		xCellEditor.setValue(value);

		return super.prepareEditor(editor, row, column);
	}

	/**
	 * Focusses on the editor
	 *
	 * @see javax.swing.JTable#editCellAt
	 */
	public boolean editCellAt(int row, int column) {
		boolean editCellAt = super.editCellAt(row, column);

		requestFocus();

		return editCellAt;
	}

	/**
	 * FIXME No other row should be selected, unless input is validate
	 **/
	public void valueChanged(ListSelectionEvent e) {
		super.valueChanged(e);
	}

	/**
	 * Selects the row being changed
	 *
	 * @see javax.swing.JTable#tableChanged
	 **/
	public void tableChanged(TableModelEvent e) {
		int row = -1;

		if (e.getType() == TableModelEvent.INSERT) {
			// Select row being inserted
			row = e.getFirstRow();
		}
		else  if (e.getType() == TableModelEvent.UPDATE) {
			if (_value != null) {
				// Select row being updated
				row = _model.getRowIndex(_value);
			}
			else {
				// Select selected row
				row = getSelectedRow();

				if (row == -1) {
					// Select row being edited
					row = getEditingRow();
				}

				if (row == -1) {
					// Select first row
					row = 0;
				}
			}
		}
		else if (e.getType() == TableModelEvent.DELETE) {
			// Select previous row, if any
			row = e.getFirstRow() -1;
		}

		super.tableChanged(e);

		if (row != -1 && row +1 <= getRowCount()) {
			setRowSelectionInterval(row, row);

			// Reset value
			_value = null;
		}
	}

	/**
	 * Focusses on the editor component
	 **/
	public void requestFocus() {
		Component component = getEditorComponent();

		if (component != null) {
			component.requestFocus();
		}
		else {
			super.requestFocus();
		}
	}

	/**
	 * Gets the XTableModel
	 *
	 * @return XTableModel
	 **/
	public XTableModel getXTableModel() {
		return _model;
	}

	/**
	 * Gets the identifying value of the selected row.
	 * @return The identifier, or null if no row is selected.
	 **/
	public String getIdentifierForSelectedRow() {
		int row = getSelectedRow();

		if (row == -1) {
			return null;
		}

		return (String)getValueAt(row, _model.getIdentifyingColumn());
	}

	/**
	 * Adds a new row
	 **/
	public void addNewRow() {
		int row = getSelectedRow() + 1;

		_model.addNewRow(row);

		editCellAt(row, _model.getIdentifyingColumn());
	}

	/**
	 * Removes a row
	 **/
	public abstract void removeRow();

	/**
	 * WRITEME
	 */
	public boolean stopCellEditing() {
		int row = getEditingRow();

		if (row != -1) {
			if (getCellEditor().stopCellEditing()) {
				// Do not stop cell editing if the row is new, e.g. if a new
				// user is added and the user login is valid, the password may
				// be still empty
				return !_model.rowIsNew(row);
			}

			return false;
		}

		return true;
	}

	/**
	 * Selects the previous row to the supplied row
	 *
	 * @param row Row
	 */
	public void selectPreviousRowTo(int row) {
		// Select previous row, if any
		row--;

		if (row == -1) {
			// First row was deleted, check if there are any rows left
			if(getRowCount() > 0) {
				row = 0;
			}
		}

		if (row != -1) {
			setRowSelectionInterval(row, row);
		}
	}

	protected void _showWarningDialog(String warning, String title) {
		Util.showWarningDialog(_getOwner(), warning, title);
	}

	protected Window _getOwner() {
		return Util.getOwner(this);
	}
}
