/*  Sesame - Storage and Querying architecture for RDF and RDF Schema
 *  Copyright (C) 2001-2006 Aduna
 *
 *  Contact:
 *  	Aduna
 *  	Prinses Julianaplein 14 b
 *  	3817 CS Amersfoort
 *  	The Netherlands
 *  	tel. +33 (0)33 465 99 87
 *  	fax. +33 (0)33 465 99 87
 *
 *  	http://aduna-software.com/
 *  	http://www.openrdf.org/
 *
 *  This library is free software; you can redistribute it and/or
 *  modify it under the terms of the GNU Lesser General Public
 *  License as published by the Free Software Foundation; either
 *  version 2.1 of the License, or (at your option) any later version.
 *
 *  This library is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 *  Lesser General Public License for more details.
 *
 *  You should have received a copy of the GNU Lesser General Public
 *  License along with this library; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

package org.openrdf.sesame.config.ui;

import java.util.Collections;
import java.util.Iterator;
import java.util.ListIterator;

import javax.swing.JLabel;

import org.openrdf.sesame.config.SystemConfig;
import org.openrdf.sesame.config.UserInfo;

/**
 * User XTableModel for UserTable
 *
 * @author Peter van 't Hof
 * @author Arjohn Kampman
 * @version $Revision: 1.4.4.2 $
 */
public class UserTableModel extends XTableModel {

/*---------------------------+
| Variables                  |
+---------------------------*/

	/** User id column id */
	public static final int COLUMN_ID = 0;

	/** User login column id */
	public static final int COLUMN_LOGIN = 1;

	/** User full name column id */
	public static final int COLUMN_FULL_NAME = 2;

	/** User password column id */
	public static final int COLUMN_PASSWORD = 3;

/*-----------------------------+
| Constructors                 |
+-----------------------------*/

	/**
	 * Creates a new UserTableModel with the supplied SystemConfig
	 *
	 * @param config SystemConfig
	 */
	public UserTableModel(SystemConfig config) {
		super(config, new ColumnData[] {
				new ColumnData("Id", 50, JLabel.LEFT),
				new ColumnData("Login", 100, JLabel.LEFT),
				new ColumnData("Full name", 200, JLabel.LEFT),
				new ColumnData("Password", 100, JLabel.LEFT)
			});
	}

/*--------+
| Methods |
+--------*/

	public int getIdentifyingColumn() {
		return COLUMN_LOGIN;
	}

	public Object getValueAt(int row, int column) {
		if (row < 0 || row >= getRowCount()) {
			return null;
		}

		UserData userData = (UserData)_rows.get(row);

		switch (column) {
			case COLUMN_ID:
				return new Integer(userData.getId());

			case COLUMN_LOGIN:
				return userData.getLogin();

			case COLUMN_FULL_NAME:
				return userData.getFullName();

			case COLUMN_PASSWORD:
				return userData.getPassword();
		}

		return null;
	}

	public void setValueAt(Object value, int row, int column) {
		if (row < 0 || row >= getRowCount()) {
			return;
		}

		UserData userData = (UserData)_rows.get(row);

		switch (column) {
			case COLUMN_ID:
				int id = 0;
				try {
					id = Integer.parseInt(value.toString());
				}
				catch (NumberFormatException e) {
					// ignore
				}

				if (userData.getId() != id) {
					// Id has changed
					_config.setUserId(userData.getLogin(), id);
				}

				break;

			case COLUMN_LOGIN:
				String newLogin = value.toString();

				if (userData.isNew()) {
					userData.setLogin(newLogin);
					fireTableRowsUpdated(row, row);
				}
				else {
					String oldLogin = userData.getLogin();

					if (!oldLogin.equals(newLogin)) {
						// Login has changed
						_config.setUserLogin(oldLogin, newLogin);
					}
				}

				break;

			case COLUMN_FULL_NAME:
				String newFullName = value.toString();

				if (userData.isNew()) {
					userData.setFullName(newFullName);
					fireTableRowsUpdated(row, row);
				}
				else if (!userData.getFullName().equals(newFullName)) {
					// Full name has changed
					_config.setUserFullName(userData.getLogin(), newFullName);
				}

				break;

			case COLUMN_PASSWORD:
				String newPassword = value.toString();

				if (userData.isNew()) {
					userData.setPassword(newPassword);

					_config.addUser(
							userData.getId(), userData.getLogin(),
							userData.getFullName(), userData.getPassword());
				}
				else if (!userData.getPassword().equals(newPassword)) {
					// Password has changed
					_config.setUserPassword(userData.getLogin(), newPassword);
				}

				break;
		}
	}

	public void configurationChanged() {
		_updateTable();
	}

	protected void _updateTable() {
		ListIterator rowsIter = _rows.listIterator();

		while (rowsIter.hasNext()) {
			UserData userData = (UserData)rowsIter.next();

			if (userData.isNew()) {
				// FIXME Table is being edited, do not update
				return;
			}
		}

		_rows.clear();

		Iterator userInfoIter = _config.getUserInfoList().iterator();

		while (userInfoIter.hasNext()) {
			UserInfo userInfo = (UserInfo)userInfoIter.next();
			_rows.add(new UserData(
					userInfo.getID(), userInfo.getLogin(),
					userInfo.getFullName(), userInfo.getPassword()));
		}

		Collections.sort(_rows);

		fireTableDataChanged();
	}

	protected RowData _createRow() {
		return new UserData(_config.getUnusedUserId());
	}

/*------------------------------------+
| Inner class UserData                |
+------------------------------------*/

	/**
	 * User data of UserTableModel
	 **/
	protected class UserData extends RowData {

	/*-------------------------+
	| Variables                |
	+-------------------------*/

		/** User id */
		protected int _id;

		/** User login */
		protected String _login;

		/** User full name */
		protected String _fullName;

		/** User password */
		protected String _password;

	/*-------------------------+
	| Constructors             |
	+-------------------------*/

		/**
		 * Creates a new UserData with the supplied user id.
		 *
		 * @param id User id
		 */
		public UserData(int id) {
			_id = id;
		}

		/**
		 * Creates a new UserData with the supplied user id, login,
		 * full name and password.
		 *
		 * @param id User id
		 * @param login User login
		 * @param fullName User full name
		 * @param password User password
		 */
		public UserData(int id, String login, String fullName, String password) {
			_id = id;
			_login = login;
			_fullName = fullName;
			_password = password;
		}

	/*--------------------------+
	| Methods                   |
	+--------------------------*/

		public String getIdentifier() {
			return getLogin();
		}

		/**
		 * Gets the user id
		 *
		 * @return User id
		 */
		public int getId() {
			return _id;
		}

		/**
		 * Gets the user login.
		 *
		 * @return A login name.
		 */
		public String getLogin() {
			return _login;
		}

		/**
		 * Sets the user login with the supplied login.
		 *
		 * @param login A login name.
		 */
		public void setLogin(String login) {
			_login = login;
		}

		/**
		 * Gets the user full name
		 *
		 * @return User full name
		 */
		public String getFullName() {
			return _fullName;
		}

		/**
		 * Sets the user full name with the supplied full name
		 *
		 * @param User full name
		 */
		public void setFullName(String fullName) {
			_fullName = fullName;
		}

		/**
		 * Gets the user password
		 *
		 * @return User password
		 */
		public String getPassword() {
			return _password;
		}

		/**
		 * Sets the user password with the supplied password
		 *
		 * @param User password
		 */
		public void setPassword(String password) {
			_password = password;
		}

		public boolean isNew() {
			return _login == null || _fullName == null || _password == null;
		}
	}
}
