/*  Sesame - Storage and Querying architecture for RDF and RDF Schema
 *  Copyright (C) 2001-2006 Aduna
 *
 *  Contact:
 *  	Aduna
 *  	Prinses Julianaplein 14 b
 *  	3817 CS Amersfoort
 *  	The Netherlands
 *  	tel. +33 (0)33 465 99 87
 *  	fax. +33 (0)33 465 99 87
 *
 *  	http://aduna-software.com/
 *  	http://www.openrdf.org/
 *
 *  This library is free software; you can redistribute it and/or
 *  modify it under the terms of the GNU Lesser General Public
 *  License as published by the Free Software Foundation; either
 *  version 2.1 of the License, or (at your option) any later version.
 *
 *  This library is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 *  Lesser General Public License for more details.
 *
 *  You should have received a copy of the GNU Lesser General Public
 *  License along with this library; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

package org.openrdf.sesame.config.ui;

import javax.swing.JOptionPane;
import javax.swing.JTextField;
import javax.swing.event.ChangeEvent;
import javax.swing.table.DefaultTableCellRenderer;
import javax.swing.table.TableCellEditor;
import javax.swing.table.TableCellRenderer;

import org.openrdf.sesame.config.SystemConfig;
import org.openrdf.sesame.config.ui.util.Util;

/**
 * User XTable of UserTab
 *
 * @author Peter van 't Hof
 * @author Arjohn Kampman
 * @version $Revision: 1.4.4.2 $
 */
public class UserTable extends XTable {

/*-------------+
| Constructors |
+-------------*/

	/**
	 * Creates a new UserTable with the supplied SystemConfig
	 *
	 * @param config SystemConfig
	 */
	public UserTable(SystemConfig config) {
		super(config);

		setXTableModel(new UserTableModel(config));
	}

/*--------+
| Methods |
+--------*/

	protected TableCellRenderer _createCellRenderer(int columnNo) {
		if (columnNo == UserTableModel.COLUMN_ID ||
			columnNo == UserTableModel.COLUMN_LOGIN ||
			columnNo == UserTableModel.COLUMN_FULL_NAME ||
			columnNo == UserTableModel.COLUMN_PASSWORD)
		{
			DefaultTableCellRenderer renderer = new DefaultTableCellRenderer();
			renderer.setHorizontalAlignment(_model.getColumnAlignment(columnNo));
			return renderer;
		}

		return null;
	}

	protected TableCellEditor _createCellEditor(int columnNo) {
		TableCellEditor editor = null;

		if (columnNo == UserTableModel.COLUMN_ID) {
			editor = new IdCellEditor(this);
		}
		else if (columnNo == UserTableModel.COLUMN_LOGIN) {
			editor = new LoginCellEditor(this);
		}
		else if (columnNo == UserTableModel.COLUMN_FULL_NAME) {
			editor = new XCellEditor(new JTextField(), this);
		}
		else if (columnNo == UserTableModel.COLUMN_PASSWORD) {
			editor = new XCellEditor(new JTextField(), this);
		}

		return editor;
	}

	/**
	 * If the editing row is new, continues editing
	 *
	 * @see javax.swing.JTable#editingStopped
	 */
	public void editingStopped(ChangeEvent e) {
		int row = getEditingRow();

 		super.editingStopped(e);

		if (_model.rowIsNew(row)) {
			if (_model.valueIsNew(row, UserTableModel.COLUMN_FULL_NAME)) {
				editCellAt(row, UserTableModel.COLUMN_FULL_NAME);
			}
			else if (_model.valueIsNew(row, UserTableModel.COLUMN_PASSWORD)) {
				editCellAt(row, UserTableModel.COLUMN_PASSWORD);
			}
		}
	}

	public void removeRow() {
		String login = getIdentifierForSelectedRow();

		if (login == null) {
			_showWarningDialog("No user selected.", "Remove User");
			return;
		}

		if (Util.showYesNoDialog(_getOwner(),
				"Are you sure you want to remove user '" + login + "'?",
				"Remove User") == JOptionPane.YES_OPTION)
		{
			int row = getSelectedRow();
			_config.removeUser(login);
			selectPreviousRowTo(row);
		}
	}

/*---------------------------------------+
| Inner class IdCellEditor               |
+---------------------------------------*/

	class IdCellEditor extends XCellEditor {

		public IdCellEditor(UserTable userTable) {
			super(new IntTextField(), userTable);
		}

		public boolean isValid() {
			IntTextField text = (IntTextField)getComponent();
			Integer integerValue = (Integer)_value;
			int oldId = integerValue.intValue();
			int newId = text.getInt();

			if (oldId != newId && (newId == 1 || newId == 2)) {
				// User id has changed
				_showWarningDialog(
					"User ID's 1 and 2 are reserved for the ADMIN and\n" +
					 "ANONYMOUS accounts respectively. Create these\n" +
					 "accounts only if you need them.",
					 "Edit User");
			}

			return true;
		}
	}

/*---------------------------------------+
| Inner class LoginCellEditor               |
+---------------------------------------*/

	class LoginCellEditor extends XCellEditor {

		public LoginCellEditor(UserTable userTable) {
			super(new JTextField(), userTable);
		}

		public boolean isValid() {
			String newLogin = getCellEditorValue().toString();

			if (newLogin.length() == 0) {
				_showWarningDialog("Login required.", "Edit User");
				return false;
			}

			String oldLogin = (String)_value;

			if (!newLogin.equals(oldLogin) && _config.hasUser(newLogin)) {
				// Login has changed
				_showWarningDialog(newLogin + " already exists.", "Edit User");
				return false;
			}

			return true;
		}
	}
}
