/*  Sesame - Storage and Querying architecture for RDF and RDF Schema
 *  Copyright (C) 2001-2006 Aduna
 *
 *  Contact:
 *  	Aduna
 *  	Prinses Julianaplein 14 b
 *  	3817 CS Amersfoort
 *  	The Netherlands
 *  	tel. +33 (0)33 465 99 87
 *  	fax. +33 (0)33 465 99 87
 *
 *  	http://aduna-software.com/
 *  	http://www.openrdf.org/
 *
 *  This library is free software; you can redistribute it and/or
 *  modify it under the terms of the GNU Lesser General Public
 *  License as published by the Free Software Foundation; either
 *  version 2.1 of the License, or (at your option) any later version.
 *
 *  This library is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 *  Lesser General Public License for more details.
 *
 *  You should have received a copy of the GNU Lesser General Public
 *  License along with this library; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

package org.openrdf.sesame.config.ui;

import java.awt.GridBagConstraints;
import java.awt.GridBagLayout;
import java.awt.event.FocusEvent;
import java.awt.event.FocusListener;
import java.awt.event.ItemEvent;
import java.awt.event.ItemListener;
import java.net.URL;

import javax.swing.BorderFactory;
import javax.swing.ImageIcon;
import javax.swing.JCheckBox;
import javax.swing.JComboBox;
import javax.swing.JLabel;
import javax.swing.JPanel;
import javax.swing.JTextArea;
import javax.swing.JTextField;

import org.openrdf.sesame.config.SystemConfig;
import org.openrdf.sesame.config.SystemConfigListener;
import org.openrdf.sesame.config.ui.util.GridBagUtil;
import org.openrdf.sesame.config.ui.util.Util;

/**
 * Server JPanel of ConfigureSesameFrame
 *
 * @author Arjohn Kampman
 * @version $Revision: 1.4.4.2 $
 */
public class ServerTab
	extends JPanel
	implements SystemConfigListener, ItemListener, FocusListener {

/*----------+
| Variables |
+----------*/

	/** SystemConfig */
	protected SystemConfig _config;

	protected JTextField _adminPassword;
	protected JTextField _logDir;
	protected JComboBox _logLevel;
	protected JTextField _tmpDir;
	protected JCheckBox _rmiEnabled;
	protected JLabel _rmiFactoryClassLabel;
	protected JTextField _rmiFactoryClass;
	protected JLabel _rmiPortLabel;
	protected IntTextField _rmiPort;

/*-------------+
| Constructors |
+-------------*/

	/**
	 * Creates a new ServerTab with the supplied SystemConfig
	 *
	 * @param config SystemConfig
	 */
	public ServerTab(SystemConfig config) {
		_config = config;
		_config.addListener(this);

		setLayout(new GridBagLayout());

		// Add an empty border of 8 pixels
		setBorder(BorderFactory.createEmptyBorder(8, 8, 8, 8));

		// Icon
		URL url = getClass().getResource("icons/server.png");
		JLabel icon = new JLabel(new ImageIcon(url));

		GridBagUtil.constrain(this, icon,
			0, 0, 1, 1,
			GridBagConstraints.NONE, GridBagConstraints.CENTER, 0, 0,
			0, 0, 4, 4);

		// Explanation
		JTextArea explanation = Util.createReadOnlyTextArea(
			"Use the fields below to change the server settings.",
			getBackground());

		GridBagUtil.constrain(this, explanation,
			1, 0, 2, 1,
			GridBagConstraints.NONE, GridBagConstraints.WEST, 0, 0,
			0, 4, 4, 0);

		// Admin password
		GridBagUtil.constrain(this, new JLabel("Admin password:"),
			0, 1, 2, 1,
			GridBagConstraints.NONE, GridBagConstraints.WEST, 0, 0,
			4, 0, 0, 0);

		_adminPassword = new JTextField();
		_adminPassword.addFocusListener(this);
		GridBagUtil.constrain(this, _adminPassword,
			2, 1, 1, 1,
			GridBagConstraints.HORIZONTAL, GridBagConstraints.WEST, 1, 0,
			4, 4, 0, 0);

		// Log directory
		GridBagUtil.constrain(this, new JLabel("Log directory:"),
			0, 2, 2, 1,
			GridBagConstraints.NONE, GridBagConstraints.WEST, 0, 0,
			20, 0, 0, 0);

		_logDir = new JTextField();
		_logDir.addFocusListener(this);
		GridBagUtil.constrain(this, _logDir,
			2, 2, 1, 1,
			GridBagConstraints.HORIZONTAL, GridBagConstraints.WEST, 1, 0,
			20, 4, 0, 0);

		// Log level
		GridBagUtil.constrain(this, new JLabel("Log level:"),
			0, 3, 2, 1,
			GridBagConstraints.NONE, GridBagConstraints.WEST, 0, 0,
			4, 0, 0, 0);

		_logLevel = new JComboBox(new String[] {
				"All", "Trace", "Status", "Warning", "Error", "None"});
		_logLevel.addItemListener(this);
		GridBagUtil.constrain(this, _logLevel,
			2, 3, 1, 1,
			GridBagConstraints.NONE, GridBagConstraints.WEST, 0, 0,
			4, 4, 0, 0);

		// Tmp directory
		GridBagUtil.constrain(this, new JLabel("Tmp directory:"),
			0, 4, 2, 1,
			GridBagConstraints.NONE, GridBagConstraints.WEST, 0, 0,
			20, 0, 0, 0);

		_tmpDir = new JTextField();
		_tmpDir.addFocusListener(this);
		GridBagUtil.constrain(this, _tmpDir,
			2, 4, 1, 1,
			GridBagConstraints.HORIZONTAL, GridBagConstraints.WEST, 1, 0,
			20, 4, 0, 0);

		// RMI service
		_rmiEnabled = new JCheckBox("RMI service enabled");
		_rmiEnabled.addItemListener(this);
		GridBagUtil.constrain(this, _rmiEnabled,
			0, 5, 3, 1,
			GridBagConstraints.HORIZONTAL, GridBagConstraints.WEST, 1, 0,
			20, 0, 0, 0);

		_rmiFactoryClassLabel = new JLabel("RMI Factory class:");
		GridBagUtil.constrain(this, _rmiFactoryClassLabel,
			0, 6, 2, 1,
			GridBagConstraints.NONE, GridBagConstraints.WEST, 0, 0,
			4, 0, 0, 0);

		_rmiFactoryClass = new JTextField();
		_rmiFactoryClass.addFocusListener(this);
		GridBagUtil.constrain(this, _rmiFactoryClass,
			2, 6, 1, 1,
			GridBagConstraints.HORIZONTAL, GridBagConstraints.WEST, 1, 0,
			4, 4, 0, 0);

		// RMI port
		_rmiPortLabel = new JLabel("RMI port:");
		GridBagUtil.constrain(this, _rmiPortLabel,
			0, 7, 2, 1,
			GridBagConstraints.NONE, GridBagConstraints.WEST, 0, 0,
			4, 0, 0, 0);

		_rmiPort = new IntTextField();
		_rmiPort.addFocusListener(this);
		GridBagUtil.constrain(this, _rmiPort,
			2, 7, 1, 1,
			GridBagConstraints.HORIZONTAL, GridBagConstraints.WEST, 1, 0,
			4, 4, 0, 0);

		// Fill empty space
		GridBagUtil.constrain(this, new JPanel(),
			0, 8, 3, 1,
			GridBagConstraints.BOTH, GridBagConstraints.CENTER, 1, 1,
			0, 0, 0, 0);

		_updateTab();
	}

	/**
	 * Updates the tab (sets the values of text fields, combo boxes, etc.)
	 **/
	protected void _updateTab() {
		// Admin password
		if (_config.getAdminPassword() == null) {
			_adminPassword.setText("");
		}
		else {
			_adminPassword.setText(_config.getAdminPassword());
		}

		// log directory
		if (_config.getLogDir() == null) {
			_logDir.setText("");
		}
		else {
			_logDir.setText(_config.getLogDir());
		}

		// Log level
		_logLevel.setSelectedIndex(5 - _config.getLogLevel());

		// tmp directory
		if (_config.getTmpDir() == null) {
			_tmpDir.setText("");
		}
		else {
			_tmpDir.setText(_config.getTmpDir());
		}

		// RMI service
		boolean rmiEnabled = _config.isRMIEnabled();
		_rmiEnabled.setSelected(rmiEnabled);
		_rmiFactoryClass.setText(_config.getRMIFactoryClass());
		_rmiPort.setInt(_config.getRMIPort());

		_rmiFactoryClassLabel.setEnabled(rmiEnabled);
		_rmiFactoryClass.setEnabled(rmiEnabled);
		_rmiPortLabel.setEnabled(rmiEnabled);
		_rmiPort.setEnabled(rmiEnabled);
	}

/*---------------------------------------------+
| Methods from interface SystemConfigListener  |
+---------------------------------------------*/

	public void configurationChanged() {
		_updateTab();
	}

/*---------------------------------------------+
| Methods from interface FocusListener         |
+---------------------------------------------*/

	public void focusGained(FocusEvent e) {
		// ignore
	}

	public void focusLost(FocusEvent e) {
		Object source = e.getSource();

		if (source == _adminPassword) {
			String newAdminPassword = _adminPassword.getText();
			if (!newAdminPassword.equals(_config.getAdminPassword())) {
				_config.setAdminPassword(newAdminPassword);
			}
		}
		else if (source == _logDir) {
			String newLogDir = _logDir.getText();
			if (newLogDir.trim().length() == 0) {
				if (_config.getLogDir() != null) {
					_config.setLogDir(null);
				}
			}
			else {
				if (!newLogDir.equals(_config.getLogDir())) {
					_config.setLogDir(newLogDir);
				}
			}
		}
		else if (source == _tmpDir) {
			String newTmpDir = _tmpDir.getText();
			if (newTmpDir.trim().length() == 0) {
				if (_config.getTmpDir() != null) {
					_config.setTmpDir(null);
				}
			}
			else {
				if (!newTmpDir.equals(_config.getTmpDir())) {
					_config.setTmpDir(newTmpDir);
				}
			}
		}
	}

/*---------------------------------------------+
| Methods from interface ItemListener          |
+---------------------------------------------*/

	public void itemStateChanged(ItemEvent e) {
		Object source = e.getSource();

		if (source == _logLevel) {
			int newLogLevel = 5 - _logLevel.getSelectedIndex();
			if (newLogLevel != _config.getLogLevel()) {
				_config.setLogLevel(newLogLevel);
			}
		}
		else if (source == _rmiEnabled) {
			_config.setRMIEnabled(_rmiEnabled.isSelected());
		}
	}
}
