package org.openrdf.sesame.config.ui;

import java.awt.Container;
import java.awt.Font;
import java.awt.GridBagConstraints;
import java.awt.GridBagLayout;
import java.awt.Insets;
import java.awt.event.ActionEvent;
import java.awt.event.ActionListener;
import java.awt.event.KeyEvent;
import java.awt.event.KeyListener;
import java.io.FileNotFoundException;
import java.io.IOException;
import java.io.InputStream;
import java.io.StringWriter;
import java.net.ConnectException;
import java.net.HttpURLConnection;
import java.net.MalformedURLException;
import java.net.URL;
import java.net.UnknownHostException;
import java.util.HashMap;
import java.util.Map;

import javax.swing.JButton;
import javax.swing.JComponent;
import javax.swing.JDialog;
import javax.swing.JFrame;
import javax.swing.JLabel;
import javax.swing.JOptionPane;
import javax.swing.JPanel;
import javax.swing.JPasswordField;
import javax.swing.JSeparator;
import javax.swing.JTextField;

import org.openrdf.util.http.HttpClientUtil;

import org.openrdf.sesame.config.SystemConfig;
import org.openrdf.sesame.config.handlers.SystemConfigFileHandler;

class ServerIOPanel extends JPanel implements ActionListener, KeyListener {

/*--------------------------------------+
| Constants                             |
+--------------------------------------*/

	public static final int LOAD = 1;
	public static final int STORE = 2;

/*--------------------------------------+
| Variables                             |
+--------------------------------------*/

	// Panel components
	protected JTextField _serverUrlField;
	protected JPasswordField _passwordField;
	protected JLabel _storeNote;
	protected JButton _actionButton;
	protected JButton _cancelButton;

	protected int _mode;

	protected JDialog _dialog;

	protected SystemConfig _config;

/*--------------------------------------+
| Constructors                          |
+--------------------------------------*/

	/**
	 * Creates a new ServerIOPanel that is in LOAD mode by default.
	 *
	 * @see #setMode(int)
	 **/
	public ServerIOPanel() {
		_mode = LOAD;

		_serverUrlField = new JTextField(30);
		_passwordField = new JPasswordField(10);

		_actionButton = new JButton("Load");
		_cancelButton = new JButton("Cancel");

		_storeNote = new JLabel("Note: any changes will be effective immediately.");
		Font font = _storeNote.getFont().deriveFont(Font.PLAIN);
		_storeNote.setFont(font);
		// _storeNote is invisible in LOAD mode
		_storeNote.setVisible(false);

		// Use GridBagLayout to layout components
		GridBagLayout gridBag = new GridBagLayout();
		this.setLayout(gridBag);

		// Add components to content pane
		_addComponent(this, gridBag, new JLabel("Server URL:"),
				0, 0, 2, 1,
				0, 0, GridBagConstraints.WEST, GridBagConstraints.NONE,
				4, 8, 0, 8);

		_addComponent(this, gridBag, _serverUrlField,
				0, 1, 2, 1,
				1, 0, GridBagConstraints.WEST, GridBagConstraints.HORIZONTAL,
				0, 8, 0, 8);

		_addComponent(this, gridBag, new JLabel("(Current) admin password:"),
				0, 2, 1, 1,
				0, 0, GridBagConstraints.WEST, GridBagConstraints.NONE,
				8, 8, 0, 4);

		_addComponent(this, gridBag, _passwordField,
				1, 2, 1, 1,
				1, 0, GridBagConstraints.WEST, GridBagConstraints.HORIZONTAL,
				8, 0, 0, 8);

		_addComponent(this, gridBag, _storeNote,
				0, 3, 2, 1,
				1, 0, GridBagConstraints.WEST, GridBagConstraints.NONE,
				8, 8, 0, 8);

		_addComponent(this, gridBag, new JSeparator(),
				0, 4, 2, 1,
				1, 0, GridBagConstraints.CENTER, GridBagConstraints.HORIZONTAL,
				8, 8, 0, 8);

		_addComponent(this, gridBag, _cancelButton,
				0, 5, 1, 1,
				0, 0, GridBagConstraints.WEST, GridBagConstraints.NONE,
				8, 8, 8, 4);

		_addComponent(this, gridBag, _actionButton,
				1, 5, 1, 1,
				0, 0, GridBagConstraints.EAST, GridBagConstraints.NONE,
				8, 4, 8, 8);

		// Listen to actions on the buttons
		_actionButton.addActionListener(this);
		_cancelButton.addActionListener(this);

		// Listen for keys
		_actionButton.addKeyListener(this);
		_serverUrlField.addKeyListener(this);
		_passwordField.addKeyListener(this);

		// Insert default server URL
		_serverUrlField.setText("http://localhost:8080/sesame/");
		_serverUrlField.selectAll();
		_serverUrlField.requestFocus();
	}

/*--------------------------------------+
| Methods                               |
+--------------------------------------*/

	public String getServerURL() {
		return _serverUrlField.getText().trim();
	}

	public SystemConfig showLoadDialog(JFrame parent) {
		setMode(LOAD);

		_showDialog(parent);

		return _config;
	}

	public void showSendDialog(JFrame parent, SystemConfig config) {
		_config = config;

		setMode(STORE);

		_showDialog(parent);
	}

	protected void _showDialog(JFrame parent) {
		if (_dialog != null) {
			// Hide existing dialog first
			hideDialog();
		}

		_dialog = new JDialog(parent, true);
		Container cpane = _dialog.getContentPane();
		cpane.add(this);

		if (_mode == LOAD) {
			_dialog.setTitle("Load from Server");
		}
		else {
			_dialog.setTitle("Send to Server");
		}

		_dialog.pack();

		_dialog.setLocationRelativeTo(parent);

		_dialog.setVisible(true);
	}

	public void hideDialog() {
		if (_dialog != null) {
			_dialog.setVisible(false);
			_dialog.dispose();
			_dialog = null;
		}
	}

	/**
	 * Sets the mode of this panel. The panel can operate in two modes: LOAD
	 * and STORE. In LOAD mode, the panel will ask the user to specify info
	 * for loading data. In STORE mode, the panel will ask the user to specify
	 * info for storing data.
	 *
	 * @see #LOAD
	 * @see #STORE
	 **/
	public void setMode(int mode) {
		if (mode != LOAD && mode != STORE) {
			throw new IllegalArgumentException("Unknown mode");
		}

		if (_mode != mode) {
			_mode = mode;

			_storeNote.setVisible( _mode == STORE );

			_switchModes();
		}
	}

	/**
	 * Switches the panel to the mode currently indicated by the _mode
	 * variable.
	 **/
	protected void _switchModes() {
		if (_mode == LOAD) {
			_actionButton.setText("Load");
		}
		else {
			_actionButton.setText("Send");
		}
	}

	public void actionPerformed(ActionEvent e) {
		if (e.getSource() == _actionButton) {
			if (_mode == LOAD) {
				boolean success = _loadData();
				if (success) {
					hideDialog();
				}
			}
			else { // _mode == STORE
				boolean success = _storeData();
				if (success) {
					hideDialog();
				}
			}
		}
		else if (e.getSource() == _cancelButton) {
			hideDialog();
		}
	}

	public void keyPressed(KeyEvent e){
		if (e.getKeyCode() == KeyEvent.VK_ENTER) {
			if (_mode == LOAD) {
				boolean success = _loadData();
				if (success) {
					hideDialog();
				}
			}
			else { // _mode == STORE
				boolean success = _storeData();
				if (success) {
					hideDialog();
				}
			}
		}
	}

	public void keyReleased(KeyEvent e) {
	}

	public void keyTyped(KeyEvent e) {
	}

	private void _addComponent(
			Container cont, GridBagLayout gridBag, JComponent comp,
			int x, int y, int width, int height,
			int weightX, int weightY, int anchor, int fill,
			int padTop, int padLeft, int padBottom, int padRight)
	{
		GridBagConstraints gbc = new GridBagConstraints(
				x, y, width, height,
				weightX, weightY, anchor, fill,
				new Insets(padTop,padLeft,padBottom,padRight), 0, 0);

		gridBag.setConstraints(comp, gbc);
		cont.add(comp);
	}

	protected boolean _loadData() {
		// Get parameters
		String serverURL = _serverUrlField.getText().trim();
		String password = new String(_passwordField.getPassword()).trim();

		// Check parameters
		if (serverURL.length() == 0) {
			_showInputError("Please specify a server URL");
			return false;
		}

		// Prepare request
		if (!serverURL.endsWith("/")) {
			serverURL += "/";
		}
		serverURL += "servlets/config";

		Map postParams = new HashMap();
		postParams.put("action", "getConfig");
		if (password.length() > 0) {
			postParams.put("password", password);
		}

		// Get the configuration file
		try {
			// Prepare call to server
			URL url = new URL(serverURL);
			HttpURLConnection conn = (HttpURLConnection)url.openConnection();
			HttpClientUtil.prepareMultipartPostRequest(conn, postParams, "UTF-8");

			// Execute call
			conn.connect();

			// Check response code
			int responseCode = conn.getResponseCode();
			if (responseCode != 200) {
				String responseMsg = conn.getResponseMessage();
				_showCommunicationError("The server returned code " + responseCode + ": " + responseMsg);
				return false;
			}

			InputStream resultStream = conn.getInputStream();
			_config = SystemConfigFileHandler.readConfiguration(resultStream);
			resultStream.close();

			// Disconnect
			conn.disconnect();

			return true;
		}
		catch (MalformedURLException e) {
			_showInputError("The specified server URL is not a legal URL");
			return false;
		}
		catch (IOException e) {
			_showCommunicationError(_createErrorMessage(e));
			return false;
		}
	}

	protected boolean _storeData() {
		// Get parameters
		String serverURL = _serverUrlField.getText().trim();
		String password = new String(_passwordField.getPassword()).trim();

		// Check parameters
		if (serverURL.length() == 0) {
			_showInputError("Please specify a server URL");
			return false;
		}

		// Prepare request
		if (!serverURL.endsWith("/")) {
			serverURL += "/";
		}
		serverURL += "servlets/config";

		try {
			// Get String-representation of SystemConfig
			StringWriter stringWriter = new StringWriter(4096);
			SystemConfigFileHandler.writeConfiguration(_config, stringWriter);
			stringWriter.close();
			String config = stringWriter.toString();

			// Build post data string
			Map postParams = new HashMap();
			postParams.put("action", "setConfig");
			if (password.length() > 0) {
				postParams.put("password", password);
			}
			postParams.put("config", config);

			// Prepare call to server
			URL url = new URL(serverURL);
			HttpURLConnection conn = (HttpURLConnection)url.openConnection();
			HttpClientUtil.prepareMultipartPostRequest(conn, postParams, "UTF-8");

			// Execute call
			conn.connect();

			// Get responseCode
			int responseCode = conn.getResponseCode();

			// Disconnect
			conn.disconnect();

			// Check responseCode
			if (responseCode != 200) {
				String responseMsg = conn.getResponseMessage();
				_showCommunicationError("The server returned code " + responseCode + ": " + responseMsg);
				return false;
			}

			JOptionPane.showMessageDialog(this,
					"The configuration was sent successfully",
					"Configuration sent", JOptionPane.INFORMATION_MESSAGE);
			return true;
		}
		catch (MalformedURLException e) {
			_showInputError("The specified server URL is not a legal URL");
			return false;
		}
		catch (IOException e) {
			_showCommunicationError(_createErrorMessage(e));
			return false;
		}
	}

	protected void _showInputError(String msg) {
		JOptionPane.showMessageDialog(this, msg,
				"Input error", JOptionPane.ERROR_MESSAGE);
	}

	protected void _showCommunicationError(String msg) {
		JOptionPane.showMessageDialog(this, msg,
				"Communication error", JOptionPane.ERROR_MESSAGE);
	}

	/**
	 * Creates an error messages from an IOException that is returned by
	 * the SesameClient class.
	 **/
	protected String _createErrorMessage(IOException e) {
		if (e instanceof UnknownHostException) {
			return "Unknown host: " + e.getMessage();
		}
		else if (e instanceof ConnectException) {
			return "Connection refused by host";
		}
		else if (e instanceof FileNotFoundException) {
			return "Unable to find required Sesame service";
		}
		else {
			return e.getMessage();
		}
	}
}
