/*  Sesame - Storage and Querying architecture for RDF and RDF Schema
 *  Copyright (C) 2001-2006 Aduna
 *
 *  Contact:
 *  	Aduna
 *  	Prinses Julianaplein 14 b
 *  	3817 CS Amersfoort
 *  	The Netherlands
 *  	tel. +33 (0)33 465 99 87
 *  	fax. +33 (0)33 465 99 87
 *
 *  	http://aduna-software.com/
 *  	http://www.openrdf.org/
 *
 *  This library is free software; you can redistribute it and/or
 *  modify it under the terms of the GNU Lesser General Public
 *  License as published by the Free Software Foundation; either
 *  version 2.1 of the License, or (at your option) any later version.
 *
 *  This library is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 *  Lesser General Public License for more details.
 *
 *  You should have received a copy of the GNU Lesser General Public
 *  License along with this library; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

package org.openrdf.sesame.config.ui;

import java.util.Iterator;

import javax.swing.JLabel;

import org.openrdf.sesame.config.SailConfig;
import org.openrdf.sesame.config.SystemConfig;

/**
 * Sail XTableModel for SailTable
 *
 * @author Peter van 't Hof
 * @author Arjohn Kampman
 * @version $Revision: 1.4.4.2 $
 */
public class SailTableModel extends XTableModel {

/*--------------------------+
| Variables                 |
+--------------------------*/

	/** Sail id column id */
	public static final int COLUMN_CLASS = 0;

	/** Repository id */
	protected String _id;

/*--------------------------+
| Constructors              |
+--------------------------*/

	/**
	 * Creates a new SailTableModel with the supplied repository id and
	 * SystemConfig
	 *
	 * @param id Repository id
	 * @param config SystemConfig
	 */
	public SailTableModel(String id, SystemConfig config) {
		super(config, new ColumnData[] {
				new ColumnData("Class", 133, JLabel.LEFT)
			});

		_id = id;

		_updateTable();
	}

/*--------+
| Methods |
+--------*/

	public int getIdentifyingColumn() {
		return COLUMN_CLASS;
	}

	public Object getValueAt(int row, int column) {
		if (row < 0 || row >= getRowCount()) {
			return null;
		}

		SailData sail = (SailData)_rows.get(row);

		switch (column) {
			case COLUMN_CLASS:
				return sail.getSailClass();
		}

		return null;
	}

	public void setValueAt(Object value, int row, int column) {
		if (row < 0 || row >= getRowCount()) {
			return;
		}

		SailData sailData = (SailData)_rows.get(row);
		String stringValue = value.toString();

		switch (column) {
			case COLUMN_CLASS:
				if (sailData.isNew()) {
					sailData.setSailClass(stringValue);
					_config.addSail(_id, sailData.getSailClass(), row);
				}
				else {
					String sailClass = sailData.getSailClass();

					if (!sailClass.equals(stringValue)) {
						// Class has changed
						_config.setSail(_id, sailClass, stringValue);
					}
				}

				break;
		}
	}

	public void configurationChanged() {
		_updateTable();
	}

	protected void _updateTable() {
		Iterator rowsIter = _rows.iterator();

		while (rowsIter.hasNext()) {
			SailData sailData = (SailData)rowsIter.next();

			if (sailData.isNew()) {
				// FIXME Table is being edited, do not update
				return;
			}
		}

		_rows.clear();

		if (!_config.hasRepository(_id)) {
			// Repository does not exist anymore
			_id = null;
		}
		else {
			Iterator sailInfoIter = _config.getSails(_id).iterator();

			while (sailInfoIter.hasNext()) {
				SailConfig sailConfig = (SailConfig)sailInfoIter.next();
				_rows.add(new SailData(sailConfig.getSailClass()));
			}
		}

		fireTableDataChanged();
	}

	protected RowData _createRow() {
		return new SailData();
	}

/*---------------------------+
| Inner class SailData       |
+---------------------------*/

	/**
	 * Sail data of SailTableModel
	 **/
	protected class SailData extends RowData {

	/*--------------------------+
	| Variables                 |
	+--------------------------*/

		/** Class */
		protected String _sailClass;

	/*--------------------------+
	| Constructors              |
	+--------------------------*/

		/**
		 * Creates a new SailData
		 **/
		public SailData() {
		}

		/**
		 * Creates a new SailData with the supplied class
		 *
		 * @param sailClass Class
		 */
		public SailData(String sailClass) {
			_sailClass = sailClass;
		}

	/*-------------------------+
	| Methods                  |
	+-------------------------*/

		public String getIdentifier() {
			return getSailClass();
		}

		/**
		 * Gets the class
		 *
		 * @return Class
		 */
		public String getSailClass() {
			return _sailClass;
		}

		/**
		 * Sets the class with the supplied class
		 *
		 * @param sailClass Class
		 */
		public void setSailClass(String sailClass) {
			_sailClass = sailClass;
		}

		public boolean isNew() {
			return _sailClass == null;
		}
	}
}
