/*  Sesame - Storage and Querying architecture for RDF and RDF Schema
 *  Copyright (C) 2001-2006 Aduna
 *
 *  Contact:
 *  	Aduna
 *  	Prinses Julianaplein 14 b
 *  	3817 CS Amersfoort
 *  	The Netherlands
 *  	tel. +33 (0)33 465 99 87
 *  	fax. +33 (0)33 465 99 87
 *
 *  	http://aduna-software.com/
 *  	http://www.openrdf.org/
 *
 *  This library is free software; you can redistribute it and/or
 *  modify it under the terms of the GNU Lesser General Public
 *  License as published by the Free Software Foundation; either
 *  version 2.1 of the License, or (at your option) any later version.
 *
 *  This library is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 *  Lesser General Public License for more details.
 *
 *  You should have received a copy of the GNU Lesser General Public
 *  License along with this library; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

package org.openrdf.sesame.config.ui;

import javax.swing.JOptionPane;
import javax.swing.JTextField;
import javax.swing.table.DefaultTableCellRenderer;
import javax.swing.table.TableCellEditor;
import javax.swing.table.TableCellRenderer;

import org.openrdf.sesame.config.SystemConfig;
import org.openrdf.sesame.config.ui.util.Util;

/**
 * Sail XTable of RepositoryTab
 *
 * @author Peter van 't Hof
 * @author Arjohn Kampman
 * @version $Revision: 1.4.4.2 $
 */
public class SailTable extends XTable {

/*----------+
| Variables |
+----------*/

	/** Repository id */
	protected String _id;

/*-------------+
| Constructors |
+-------------*/
	/**
	 * Creates a new SailTable with the supplied repository id and SystemConfig
	 *
	 * @param id Repository id
	 * @param config SystemConfig
	 */
	public SailTable(String id, SystemConfig config) {
		super(config);
		_id = id;

		setXTableModel(new SailTableModel(id, config));
	}

/*--------+
| Methods |
+--------*/

	protected TableCellRenderer _createCellRenderer(int columnNo) {
		if (columnNo == SailTableModel.COLUMN_CLASS) {
			DefaultTableCellRenderer renderer = new DefaultTableCellRenderer();
			renderer.setHorizontalAlignment(_model.getColumnAlignment(columnNo));
			return renderer;
		}

		return null;
	}

	protected TableCellEditor _createCellEditor(int columnNo) {
		TableCellEditor editor = null;

		if (columnNo == SailTableModel.COLUMN_CLASS) {
			editor = new ClassCellEditor(this);
		}

		return editor;
	}

	public void removeRow() {
		String sailClass = getIdentifierForSelectedRow();

		if (sailClass == null) {
			_showWarningDialog("No sail selected.", "Remove Sail");
			return;
		}

		if (Util.showYesNoDialog(_getOwner(),
				"Are you sure you want to remove Sail\n" + sailClass + "\nfrom the sail stack?",
				"Remove Sail") == JOptionPane.YES_OPTION)
		{
			int row = getSelectedRow();
			_config.removeSail(_id, sailClass);
			selectPreviousRowTo(row);
		}
	}

	/**
	 * Moves the selected sail one level up on the sail stack
	 */
	public void sailUp() {
		String sailClass = getIdentifierForSelectedRow();

		if (sailClass == null) {
			_showWarningDialog("No sail selected.", "Sail Up");
			return;
		}

		_value = sailClass;
		_config.sailUp(_id, sailClass);
	}

	/**
	 * Moves the selected sail one level down on the sail stack
	 */
	public void sailDown() {
		String sailClass = getIdentifierForSelectedRow();

		if (sailClass == null) {
			_showWarningDialog("No sail selected.", "Sail Down");
			return;
		}

		_value = sailClass;
		_config.sailDown(_id, sailClass);
	}

/*------------------------------------------+
| Inner class ClassCellEditor               |
+------------------------------------------*/

	class ClassCellEditor extends XCellEditor {

		public ClassCellEditor(SailTable sailTable) {
			super(new JTextField(), sailTable);
		}

		public boolean isValid() {
			String newClass = getCellEditorValue().toString();

			if (newClass.length() == 0) {
				_showWarningDialog("Class required.", "Edit Sail");
				return false;
			}

			String oldClass = (String)_value;

			if (!newClass.equals(oldClass) && _config.hasSail(_id, newClass)) {
				// Class has changed
				_showWarningDialog(newClass + " already exists.", "Edit Sail");
				return false;
			}

			return true;
		}
	}
}
