/*  Sesame - Storage and Querying architecture for RDF and RDF Schema
 *  Copyright (C) 2001-2006 Aduna
 *
 *  Contact:
 *  	Aduna
 *  	Prinses Julianaplein 14 b
 *  	3817 CS Amersfoort
 *  	The Netherlands
 *  	tel. +33 (0)33 465 99 87
 *  	fax. +33 (0)33 465 99 87
 *
 *  	http://aduna-software.com/
 *  	http://www.openrdf.org/
 *
 *  This library is free software; you can redistribute it and/or
 *  modify it under the terms of the GNU Lesser General Public
 *  License as published by the Free Software Foundation; either
 *  version 2.1 of the License, or (at your option) any later version.
 *
 *  This library is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 *  Lesser General Public License for more details.
 *
 *  You should have received a copy of the GNU Lesser General Public
 *  License along with this library; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

package org.openrdf.sesame.config.ui;

import java.util.Collections;
import java.util.Iterator;

import javax.swing.JLabel;

import org.openrdf.sesame.config.RepositoryInfo;
import org.openrdf.sesame.config.SystemConfig;

/**
 * Repository XTableModel for RepositoryTable
 *
 * @author Peter van 't Hof
 * @author Arjohn Kampman
 * @version $Revision: 1.4.4.2 $
 */
public class RepositoryTableModel extends XTableModel {

/*------------------------------+
| Variables                     |
+------------------------------*/

	/** Column number for repository ID. */
	public static final int COLUMN_ID = 0;

	/** Column number for repository title. */
	public static final int COLUMN_TITLE = 1;

/*-------------------------------+
| Constructors                   |
+-------------------------------*/

	/**
	 * Creates a new RepositoryTableModel with the supplied SystemConfig
	 *
	 * @param config SystemConfig
	 */
	public RepositoryTableModel(SystemConfig config) {
		super(config, new ColumnData[] {
				new ColumnData("Id", 125, JLabel.LEFT),
		 		new ColumnData("Title", 325, JLabel.LEFT)
			});
	}

/*--------+
| Methods |
+--------*/

	public int getIdentifyingColumn() {
		return COLUMN_ID;
	}

	public Object getValueAt(int row, int column) {
		if (row < 0 || row >= getRowCount()) {
			return null;
		}

		RepositoryData repository = (RepositoryData)_rows.get(row);

		switch (column) {
			case COLUMN_ID:
				return repository.getId();

			case COLUMN_TITLE:
				return repository.getTitle();
		}

		return null;
	}

	public void setValueAt(Object value, int row, int column) {
		if (row < 0 || row >= getRowCount()) {
			return;
		}

		RepositoryData repository = (RepositoryData)_rows.get(row);
		String stringValue = value.toString();

		switch (column) {
			case COLUMN_ID:
				if (repository.isNew()) {
					repository.setId(stringValue);

					fireTableRowsUpdated(row, row);
				}
				else {
					String id = repository.getId();

					if (!id.equals(stringValue)) {
						// Id has changed
						_config.setRepositoryId(id, stringValue);
					}
				}

				break;

			case COLUMN_TITLE:
				if (repository.isNew()) {
					repository.setTitle(stringValue);
					_config.addRepository(repository.getId(), repository.getTitle());
				}
				else if (!repository.getTitle().equals(stringValue)) {
					// Title has changed
					_config.setRepositoryTitle(repository.getId(), stringValue);
				}

				break;
		}
	}

	public void configurationChanged() {
		_updateTable();
	}

	protected void _updateTable() {
		Iterator rowsIter = _rows.iterator();

		while (rowsIter.hasNext()) {
			RepositoryData repData = (RepositoryData)rowsIter.next();

			if (repData.isNew()) {
				// FIXME Table is being edited, do not update
				return;
			}
		}

		_rows.clear();

		Iterator repInfoIter = _config.getRepositoryConfigList().iterator();

		while (repInfoIter.hasNext()) {
			RepositoryInfo repInfo = (RepositoryInfo)repInfoIter.next();
			_rows.add(new RepositoryData(repInfo.getRepositoryId(), repInfo.getTitle()));
		}

		Collections.sort(_rows);

		fireTableDataChanged();
	}

	/**
	 * Clones the row at row index <tt>rowIdx</tt>.
	 * @return The ID of the newly cloned repository.
	 **/
	public String cloneRow(int rowIdx) {
		// Get the ID of the selected repository
		RepositoryData orig = (RepositoryData)_rows.get(rowIdx);
		String origId = orig.getId();

		// Derive a new repository ID from the original
		String cloneId = null;
		int i = 1;
		boolean isNewID = false;

		while (!isNewID) {
			i++;
			cloneId = origId + "-" + i;

			// Check if this ID is already in use:
			isNewID = true;

			Iterator rowsIter = _rows.iterator();
			while (rowsIter.hasNext()) {
				RepositoryData repData = (RepositoryData)rowsIter.next();
				if (repData.getId().equals(cloneId)) {
					isNewID = false;
					break;
				}
			}
		}

		_config.cloneRepository(origId, cloneId);

		return cloneId;
	}

	protected RowData _createRow() {
		return new RepositoryData();
	}

/*---------------------------+
| Inner class RepositoryData |
+---------------------------*/

	/** Repository data of RepositoryTableModel */
	protected class RepositoryData extends RowData {

	/*-------------------------+
	| Variables                |
	+-------------------------*/

		/** Repository id */
		protected String _id;

		/** Repository title */
		protected String _title;

	/*-------------------------+
	| Constructors             |
	+-------------------------*/

		/**
		 * Creates a new RepositoryData
		 **/
		public RepositoryData() {
		}

		/**
		 * Creates a new RepositoryData with the supplied repository id and title
		 *
		 * @param id Repository id
		 * @param title Repository title
		 */
		public RepositoryData(String id, String title) {
			_id = id;
			_title = title;
		}

	/*-------------------------+
	| Methods                  |
	+-------------------------*/

		public String getIdentifier() {
			return getId();
		}

		/**
		 * Gets the repository id
		 *
		 * @return Repository id
		 */
		public String getId() {
			return _id;
		}

		/**
		 * Gets the repository title
		 *
		 * @return Repository title
		 */
		public String getTitle() {
			return _title;
		}

		/**
		 * Sets the repository id with the supplied id
		 *
		 * @param Repository id
		 */
		public void setId(String id) {
			_id = id;
		}

		/**
		 * Sets the repository title with the supplied title
		 *
		 * @param Repository title
		 */
		public void setTitle(String title) {
			_title = title;
		}

		public boolean isNew() {
			return _id == null || _title == null;
		}
	}
}
