/*  Sesame - Storage and Querying architecture for RDF and RDF Schema
 *  Copyright (C) 2001-2006 Aduna
 *
 *  Contact:
 *  	Aduna
 *  	Prinses Julianaplein 14 b
 *  	3817 CS Amersfoort
 *  	The Netherlands
 *  	tel. +33 (0)33 465 99 87
 *  	fax. +33 (0)33 465 99 87
 *
 *  	http://aduna-software.com/
 *  	http://www.openrdf.org/
 *
 *  This library is free software; you can redistribute it and/or
 *  modify it under the terms of the GNU Lesser General Public
 *  License as published by the Free Software Foundation; either
 *  version 2.1 of the License, or (at your option) any later version.
 *
 *  This library is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 *  Lesser General Public License for more details.
 *
 *  You should have received a copy of the GNU Lesser General Public
 *  License along with this library; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

package org.openrdf.sesame.config.ui;

import javax.swing.JOptionPane;
import javax.swing.JTextField;
import javax.swing.event.ChangeEvent;
import javax.swing.table.DefaultTableCellRenderer;
import javax.swing.table.TableCellEditor;
import javax.swing.table.TableCellRenderer;

import org.openrdf.sesame.config.SystemConfig;
import org.openrdf.sesame.config.ui.util.Util;

/**
 * Repository XTable of RepositoryTab
 *
 * @author Peter van 't Hof
 * @author Arjohn Kampman
 * @version $Revision: 1.4.4.2 $
 */
public class RepositoryTable extends XTable {

/*-------------+
| Constructors |
+-------------*/

	/**
	 * Creates a new RepositoryTable with the supplied SystemConfig
	 *
	 * @param config SystemConfig
	 */
	public RepositoryTable(SystemConfig config) {
		super(config);

		setXTableModel(new RepositoryTableModel(config));
	}

/*--------+
| Methods |
+--------*/

	protected TableCellRenderer _createCellRenderer(int columnNo) {
		if (columnNo == RepositoryTableModel.COLUMN_ID ||
			columnNo == RepositoryTableModel.COLUMN_TITLE)
		{
			DefaultTableCellRenderer renderer = new DefaultTableCellRenderer();
			renderer.setHorizontalAlignment(_model.getColumnAlignment(columnNo));
			return renderer;
		}

		return null;
	}

	protected TableCellEditor _createCellEditor(int columnNo) {
		TableCellEditor editor = null;

		if (columnNo == RepositoryTableModel.COLUMN_ID) {
			editor = new IdColumnEditor(this);
		}
		else if (columnNo == RepositoryTableModel.COLUMN_TITLE) {
			editor = new XCellEditor(new JTextField(), this);
		}

		return editor;
	}

	/**
	 * If the editing row is new, continues editing
	 *
	 * @see org.openrdf.sesame.config.ui.XTable#editingStopped
	 */
	public void editingStopped(ChangeEvent e) {
		int row = getEditingRow();

 		super.editingStopped(e);

		if (_model.valueIsNew(row, RepositoryTableModel.COLUMN_TITLE)) {
			editCellAt(row, RepositoryTableModel.COLUMN_TITLE);
		}
	}

	public void cloneRow() {
		int row = getSelectedRow();
		String cloneId = ((RepositoryTableModel)_model).cloneRow(row);
		int cloneRowIdx = _model.getRowIndex(cloneId);
		editCellAt(cloneRowIdx, RepositoryTableModel.COLUMN_ID);
	}

	public void removeRow() {
		String id = getIdentifierForSelectedRow();

		if (id == null) {
			_showWarningDialog("No repository selected.", "Remove Repository");
			return;
		}

		if (Util.showYesNoDialog(_getOwner(),
				"Are you sure you want to remove repository '" + id + "'?",
			 	"Remove Repository") == JOptionPane.YES_OPTION)
		{
			int row = getSelectedRow();
			_config.removeRepository(id);
			selectPreviousRowTo(row);
		}
	}

/*--------------------------------+
| Inner class IdColumnEditor      |
+--------------------------------*/

	class IdColumnEditor extends XCellEditor {

		public IdColumnEditor(RepositoryTable repTable) {
			super(new JTextField(), repTable);
		}

		public boolean isValid() {
			String newId = getCellEditorValue().toString();

			if (newId.length() == 0) {
				_showWarningDialog("Id required.", "Edit Repository");
				return false;
			}

			String oldId = (String)_value;

			if (!newId.equals(oldId) && _config.hasRepository(newId)) {
				_showWarningDialog(newId + " already exists.", "Edit Repository");
				return false;
			}

			return true;
		}
	}
}
