/*  Sesame - Storage and Querying architecture for RDF and RDF Schema
 *  Copyright (C) 2001-2006 Aduna
 *
 *  Contact:
 *  	Aduna
 *  	Prinses Julianaplein 14 b
 *  	3817 CS Amersfoort
 *  	The Netherlands
 *  	tel. +33 (0)33 465 99 87
 *  	fax. +33 (0)33 465 99 87
 *
 *  	http://aduna-software.com/
 *  	http://www.openrdf.org/
 *
 *  This library is free software; you can redistribute it and/or
 *  modify it under the terms of the GNU Lesser General Public
 *  License as published by the Free Software Foundation; either
 *  version 2.1 of the License, or (at your option) any later version.
 *
 *  This library is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 *  Lesser General Public License for more details.
 *
 *  You should have received a copy of the GNU Lesser General Public
 *  License along with this library; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

package org.openrdf.sesame.config.ui;

import java.util.Collections;
import java.util.Iterator;
import java.util.Map;

import javax.swing.JLabel;

import org.openrdf.sesame.config.SystemConfig;

/**
 * Parameter XTableModel for ParameterTable
 *
 * @author Peter van 't Hof
 * @author Arjohn Kampman
 * @version $Revision: 1.4.4.2 $
 */
public class ParameterTableModel extends XTableModel {

/*---------------------------+
| Variables                  |
+---------------------------*/

	/** Parameter key column id */
	public static final int COLUMN_KEY = 0;

	/** Parameter value column id */
	public static final int COLUMN_VALUE = 1;

	/** Repository id */
	protected String _id;

	/** Sail class */
	protected String _sailClass;

/*----------------------------+
| Constructors                |
+----------------------------*/

	/**
	 * Creates a new ParameterTableModel with the supplied repository id, sail
	 * class and SystemConfig
	 *
	 * @param id Repository id
	 * @param config SystemConfig
	 */
	public ParameterTableModel(String id, SystemConfig config) {
		super(config, new ColumnData[] {
				new ColumnData("Key", 133, JLabel.LEFT),
			 	new ColumnData("Value", 267, JLabel.LEFT)
			});

		_id = id;

		_updateTable();
	}

/*-----------------------------+
| Methods                      |
+-----------------------------*/

	public int getIdentifyingColumn() {
		return COLUMN_KEY;
	}

	public Object getValueAt(int row, int column) {
		if (row < 0 || row >= getRowCount()) {
			return null;
		}

		ParameterData parameter = (ParameterData)_rows.get(row);

		switch (column) {
			case COLUMN_KEY:
				return parameter.getKey();

			case COLUMN_VALUE:
				return parameter.getValue();
		}

		return null;
	}

	public void setValueAt(Object value, int row, int column) {
		if (row < 0 || row >= getRowCount()) {
			return;
		}

		ParameterData parameter = (ParameterData)_rows.get(row);
		String stringValue = value.toString();

		switch (column) {
			case COLUMN_KEY:
				if (parameter.isNew()) {
					parameter.setKey(stringValue);

					fireTableRowsUpdated(row, row);
				}
				else {
					String key = parameter.getKey();

					if (!key.equals(stringValue)) {

						// Key has changed
						_config.setParameterKey(_id, _sailClass, key, stringValue);
					}
				}

				break;

			case COLUMN_VALUE:
				if (parameter.isNew()) {
					parameter.setValue(stringValue);

					_config.setParameter(_id, _sailClass, parameter.getKey(), parameter.getValue());
				}
				else if (!parameter.getValue().equals(stringValue)) {

					// Value has changed
					_config.setParameterValue(_id, _sailClass, parameter.getKey(), stringValue);
				}

				break;
		}
	}

	/**
	 * Sets the class to the supplied class
	 *
	 * @param sailClass Sail class
	 */
	public void setSailClass(String sailClass) {
		_sailClass = sailClass;

		_updateTable();
	}

	public void configurationChanged() {
		_updateTable();
	}

	protected void _updateTable() {
		Iterator rowsIter = _rows.iterator();

		while (rowsIter.hasNext()) {
			ParameterData parameter = (ParameterData)rowsIter.next();

			if (parameter.isNew()) {
				// FIXME Table is being edited, do not update
				return;
			}
		}

		_rows.clear();

		if (!_config.hasRepository(_id)) {
			// Repository does not exist anymore
			_id = null;
		}
		else if (!_config.hasSail(_id, _sailClass)) {
			// Class does not exist anymore
			_sailClass = null;
		}
		else if (_sailClass != null ) {
			Map parameters = _config.getParameters(_id, _sailClass);
			Iterator keyIter = parameters.keySet().iterator();

			while (keyIter.hasNext()) {
				String key = (String)keyIter.next();
				String value = (String)parameters.get(key);
				_rows.add(new ParameterData(key, value));
			}

			Collections.sort(_rows);
		}

		fireTableDataChanged();
	}

	protected RowData _createRow() {
		return new ParameterData();
	}

/*----------------------------------------+
| Inner class ParameterData               |
+----------------------------------------*/

	/**
	 * Parameter data of ParameterTableModel
	 **/
	protected class ParameterData extends RowData {

	/*---------------------+
	| Variables            |
	+---------------------*/

		/** Parameter key */
		protected String _key;

		/** Parameter value */
		protected String _value;

	/*---------------------+
	| Constructors         |
	+---------------------*/

		/** Creates a new ParameterData */
		public ParameterData() {}

		/**
		 * Creates a new ParameterData with the supplied parameter key and value
		 *
		 * @param key Parameter key
		 * @param value Parameter value
		 */
		public ParameterData(String key, String value) {
			_key = key;
			_value = value;
		}

	/*---------------------+
	| Methods              |
	+---------------------*/

		public String getIdentifier() {
			return getKey();
		}

		/**
		 * Gets the parameter key
		 *
		 * @return Parameter key
		 */
		public String getKey() {
			return _key;
		}

		/**
		 * Gets the parameter value
		 *
		 * @return Parameter value
		 */
		public String getValue() {
			return _value;
		}

		/**
		 * Sets the parameter key with the supplied key
		 *
		 * @param Parameter key
		 */
		public void setKey(String key) {
			_key = key;
		}

		/**
		 * Sets the parameter value with the supplied value
		 *
		 * @param Parameter value
		 */
		public void setValue(String value) {
			_value = value;
		}

		public boolean isNew() {
			return _key == null || _value == null;
		}
	}
}
