/*  Sesame - Storage and Querying architecture for RDF and RDF Schema
 *  Copyright (C) 2001-2006 Aduna
 *
 *  Contact:
 *  	Aduna
 *  	Prinses Julianaplein 14 b
 *  	3817 CS Amersfoort
 *  	The Netherlands
 *  	tel. +33 (0)33 465 99 87
 *  	fax. +33 (0)33 465 99 87
 *
 *  	http://aduna-software.com/
 *  	http://www.openrdf.org/
 *
 *  This library is free software; you can redistribute it and/or
 *  modify it under the terms of the GNU Lesser General Public
 *  License as published by the Free Software Foundation; either
 *  version 2.1 of the License, or (at your option) any later version.
 *
 *  This library is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 *  Lesser General Public License for more details.
 *
 *  You should have received a copy of the GNU Lesser General Public
 *  License along with this library; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

package org.openrdf.sesame.config.ui;

import javax.swing.JOptionPane;
import javax.swing.JTextField;
import javax.swing.event.ChangeEvent;
import javax.swing.table.DefaultTableCellRenderer;
import javax.swing.table.TableCellEditor;
import javax.swing.table.TableCellRenderer;

import org.openrdf.sesame.config.SystemConfig;
import org.openrdf.sesame.config.ui.util.Util;

/**
 * Parameter XTable of AdvancedSettingDialog
 *
 * @author Peter van 't Hof
 * @author Arjohn Kampman
 * @version $Revision: 1.4.4.2 $
 */
public class ParameterTable extends XTable {

/*----------+
| Variables |
+----------*/

	/** Repository id */
	String _id;

	/** Sail Class */
	String _sailClass;

/*-------------+
| Constructors |
+-------------*/

	/**
	 * Creates a new ParameterTable with the supplied repository id and
	 * SystemConfig
	 *
	 * @param id Repository id
	 * @param config SystemConfig
	 */
	public ParameterTable(String id, SystemConfig config) {
		super(config);
		_id = id;

		setXTableModel(new ParameterTableModel(id, config));
	}

/*--------+
| Methods |
+--------*/

	protected TableCellRenderer _createCellRenderer(int columnNo) {
		if (columnNo == ParameterTableModel.COLUMN_KEY ||
			columnNo == ParameterTableModel.COLUMN_VALUE)
		{
			DefaultTableCellRenderer renderer = new DefaultTableCellRenderer();
			renderer.setHorizontalAlignment(_model.getColumnAlignment(columnNo));
			return renderer;
		}

		return null;
	}

	protected TableCellEditor _createCellEditor(int columnNo) {
		TableCellEditor editor = null;

		if (columnNo == ParameterTableModel.COLUMN_KEY) {
			editor = new KeyCellEditor(this);
		}
		else if (columnNo == ParameterTableModel.COLUMN_VALUE) {
			editor = new XCellEditor(new JTextField(), this);
		}

		return editor;
	}

	/**
	 * If the editing row is new, continues editing
	 *
	 * @see org.openrdf.sesame.config.ui.XTable#editingStopped
	 */
	public void editingStopped(ChangeEvent e) {
		int row = getEditingRow();

 		super.editingStopped(e);

		if (_model.valueIsNew(row, ParameterTableModel.COLUMN_VALUE)) {
			editCellAt(row, ParameterTableModel.COLUMN_VALUE);
		}
	}

	/**
	 * Adds a new row if a sail is defined and selected
	 *
	 * @see org.openrdf.sesame.config.ui.XTable#addNewRow
	 */
	public void addNewRow() {
		if (!_config.hasASail(_id)) {
			_showWarningDialog("No sail defined.", "Add Parameter");
			return;
		}

		if (_sailClass == null) {
			_showWarningDialog("No sail selected.", "Add Parameter");
			return;
		}

		super.addNewRow();
	}

	public void removeRow() {
		if (!_config.hasASail(_id)) {
			_showWarningDialog("No sail defined.", "Add Parameter");
			return;
		}

		if (_sailClass == null) {
			_showWarningDialog("No sail selected.", "Add Parameter");
			return;
		}

		String key = getIdentifierForSelectedRow();

		if (key == null) {
			_showWarningDialog("No parameter selected.", "Remove Parameter");
			return;
		}

		if (Util.showYesNoDialog(_getOwner(),
				"Are you sure you want to remove parameter '" + key + "'?",
				"Remove Parameter") == JOptionPane.YES_OPTION)
		{
			int row = getSelectedRow();
			_config.removeParameter(_id, _sailClass, key);
			selectPreviousRowTo(row);
		}
	}

	/**
	 * Sets the class to the supplied class
	 *
	 * @param sailClass Class
	 */
	public void setSailClass(String sailClass) {
		_sailClass = sailClass;
		ParameterTableModel model = (ParameterTableModel)_model;
		model.setSailClass(_sailClass);
	}

/*-------------------------------------+
| Inner class KeyCellEditor            |
+-------------------------------------*/

	class KeyCellEditor extends XCellEditor {

		public KeyCellEditor(ParameterTable paramTable) {
			super(new JTextField(), paramTable);
		}

		public boolean isValid() {
			String newKey = getCellEditorValue().toString();

			if (newKey.length() == 0) {
				_showWarningDialog("Key required.", "Edit Parameter");
				return false;
			}

			String oldKey = (String)_value;

			if (!newKey.equals(oldKey) &&
				_config.hasParameter(_id, _sailClass, newKey))
			{
				// Key has changed
				_showWarningDialog(
						"Parameter '" + newKey + "' already exists.",
						"Edit Parameter");
				return false;
			}

			return true;
		}
	}
}
